<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Presupuesto_model extends CI_Model {

    function __construct() {
        $this->load->database();
    }
    
    /**
    * @author Luis Navarro - Grupo Satélite
    * @since 2018-12-13
    * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
    * @return Retorna los datos de presupuesto
    */  
    public function obtener_presupuesto_grid($where = null, $groupby = null, $orderby = null) {
    $query = $this->db->select(
        "
            psp_id,
            psp_codigo,
            psp_nombre,
            psp_total,
            psp_estado_prespuesto,
            psp_fecha_desde,
            psp_fecha_hasta,
            (
                SELECT
                SUM(dgp_monto)
                FROM
                dgp_detalle_gasto_prespuesto
                WHERE
                dgp_estado = 1                
                AND dgp_id_dps IN (
                    SELECT
                        dps_id
                    FROM
                        dps_detalle_presupuesto
                    WHERE
                        dps_id_psp = psp_id
                )
            ) AS monto_ejecutado,
            (
                psp_total - 
                    (SELECT
                        SUM(dgp_monto)
                    FROM
                        dgp_detalle_gasto_prespuesto
                    WHERE
                    dgp_estado = 1                
                    AND dgp_id_dps IN (
                        SELECT
                            dps_id
                        FROM
                            dps_detalle_presupuesto
                        WHERE
                            dps_id_psp = psp_id)
                    )
            ) AS monto_ejecutar,
            psp_pendiente_ejecutar,
            SUM((
                SELECT
                SUM(dps_presupuesto_actividad)
                FROM
                dps_detalle_presupuesto
                WHERE
                dps_id_psp = psp_id
            )) AS pres_actividades
                ", false)
            ->from('psp_presupuesto')
            //->join('art_articulo', 'art_id = def_id_art')
        ;

    if ($where != null) {
        $this->db->where($where);
    }
    
    if ($groupby != null) {
        $this->db->group_by($groupby);
    }
    
    if ($orderby != null) {
        $this->db->order_by($orderby);
    }

    return $query->get()->result_array();
    }

    /**
    * @author Luis Navarro - Grupo Satélite
    * @since 2018-12-13
    * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
    * @return Retorna los datos de los detalles del presupuesto
    */  
    public function obtener_detalle_presupuesto_grid($where = null, $groupby = null, $orderby = null) {
        //$this->db->query("SET GLOBAL group_concat_max_len=4194304;");
        $query = $this->db->select("
            *,
            (
                SELECT
                    SUM(dgp_monto)
                FROM
                    dgp_detalle_gasto_prespuesto
                WHERE
                    dgp_id_dps = dps_id
                    AND dgp_estado = 1
            ) AS monto_ejecutado,
            (
                SELECT COUNT(*) FROM dgp_detalle_gasto_prespuesto
                WHERE dgp_id_dps = dps_id
            ) AS permitir_eliminar,
            (
                SELECT
                    GROUP_CONCAT(sxp_id_suc)
                FROM
                    sxp_sucursalesxpresupuesto
                WHERE
                    sxp_id_psp = dps_id_psp
            ) sucursales,
            (
                SELECT
                    GROUP_CONCAT(
                        CONCAT_WS('#', uxa_id_usu, username)
                    )
                FROM
                    uxa_usuarioxactividad_general
                    INNER JOIN users ON id = uxa_id_usu
                WHERE
                    uxa_id_acg = dps_id_acg
            ) usuarios_posibles,
            (
                SELECT
                    GROUP_CONCAT(udp_id_usu)
                FROM
                    udp_usuarioxdetalle_presupuesto
                WHERE
                    udp_id_dps = dps_id
            ) permitidos,
            (
                SELECT
                    GROUP_CONCAT(
                        CONCAT_WS('#', atp_id_art, atp_estado)
                    )
                FROM
                    atp_articulo_detalle_presupuesto
                WHERE
                    atp_id_dps = dps_id
            ) articulos
        ", false)
        ->from('dps_detalle_presupuesto')
        ->join('ars_area_servicio', 'dps_id_ars = ars_id', 'left')
        ->join('acg_actividad_general', 'dps_id_acg = acg_id', 'left')
        ->join('ace_actividad_especifica', 'dps_id_ace = ace_id', 'left')
        ->join('fuv_fuente_verificacion', 'dps_id_fuv = fuv_id', 'left')
        ->join('fnf_fuente_financiamiento', 'dps_id_fnf = fnf_id', 'left')
    ;

    if ($where != null) {
        $this->db->where($where);
    }
    
    if ($groupby != null) {
        $this->db->group_by($groupby);
    }
    
    if ($orderby != null) {
        $this->db->order_by($orderby);
    }

    return $query->get()->result_array();
    }        
    
    /**
     * @author Luis Navarro - Grupo Satélite
     * @since 2018-12-18
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Retorna los datos de los detalles de los Gastos
     */
    public function obtener_detalle_gastos_presupuesto($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select('
            *, (
                SELECT
                    GROUP_CONCAT(
                        CONCAT_WS(
                            "#",
                            rgp_id_ret,
                            rgp_porcentaje,
                            rgp_monto
                        )
                    )
                FROM
                    rgp_retencion_gasto_presupuesto
                WHERE
                    rgp_id_dgp = dgp_id
            ) AS retenciones,
            (
                SELECT
                    GROUP_CONCAT(
                        CONCAT_WS(
                            "#",
                            idp_id_imp,
                            idp_porcentaje,
                            idp_monto
                        )
                    )
                FROM
                    idp_impuesto_detalle_presupuesto
                WHERE
                    idp_id_dgp = dgp_id
            ) AS impuestos',false)
        ->from('dgp_detalle_gasto_prespuesto')
        ->join('dps_detalle_presupuesto', 'dgp_id_dps = dps_id')
        ->join('tgp_tipo_gasto_presupuesto', 'dgp_id_tgp = tgp_id', 'left')
        ->join('suc_sucursal', 'suc_id = dgp_id_suc', 'left')
        ;
        
        if ($where != null) {
            $this->db->where($where);
        }
        
        if ($groupby != null) {
            $this->db->group_by($groupby);
        }
        
        if ($orderby != null) {
            $this->db->order_by($orderby);
        }
        
        return $query->get()->result_array();
    }
    
    /**
     * @author Luis Navarro - Grupo Satélite
     * @since 2018-12-21
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Obtener los detalles de ejecucion de gastos
     */
    public function obtener_detalle_ejecucion_gastos($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select("
                ars_id,
                dps_id,
                ars_nombre,
                acg_id,
                acg_nombre,
                ace_nombre,
                dps_objetivo,
                ace_codigo,
                acg_codigo,
                fnf_nombre,
                (
                    SELECT fnc_nombre FROM psp_presupuesto
                    JOIN fnc_financiador ON fnc_id = psp_id_fnc
                    WHERE psp_id = dps_id_psp
                ) AS financiador,
                (
                SELECT
                        SUM(dgp_monto)
                    FROM
                        dgp_detalle_gasto_prespuesto
                    WHERE dgp_id_dps = dps_id
                ) AS monto_ejecutado,
                dps_presupuesto_actividad,
                dps_aporte_externo,
                dps_aporte_interno,
                fuv_nombre
            ", false)
        ->from('dps_detalle_presupuesto')
        ->join('psp_presupuesto', 'psp_id = dps_id_psp')
        ->join('ars_area_servicio', 'ars_id = dps_id_ars', 'left')
        ->join('acg_actividad_general', 'acg_id = dps_id_acg', 'left')
        ->join('ace_actividad_especifica', 'ace_id = dps_id_ace')
        ->join('fuv_fuente_verificacion', 'fuv_id = dps_id_fuv', 'left')
        ->join('fnf_fuente_financiamiento', 'fnf_id = dps_id_fnf', 'left')
        ;
        
        if ($where != null) {
            $this->db->where($where);
        }
        
        if ($groupby != null) {
            $this->db->group_by($groupby);
        }
        
        if ($orderby != null) {
            $this->db->order_by($orderby);
        }
        
        return $query->get()->result_array();
    }
    
    /**
     * @author Luis Navarro - Grupo Satélite
     * @since 2018-12-21
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Obtener los Gastos por Detalle de presupuesto 
     */
    public function reporte_detalle_gastos_ejecucion($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select("*,
                CONCAT_WS(
                    '_',
                    dps_id,
                    DATE_FORMAT(dgp_fecha, '%m'),
                    DATE_FORMAT(dgp_fecha, '%Y')
                ) clave,
            
                MONTH (dgp_fecha) AS nmes,
                DATE_FORMAT(dgp_fecha, '%M') AS mes,
                DATE_FORMAT(dgp_fecha, '%Y') AS anio,
                IFNULL(SUM(dgp_monto), 0) AS suma
            ", false)
            ->from('dgp_detalle_gasto_prespuesto')
            ->join('dps_detalle_presupuesto', 'dgp_id_dps = dps_id')
            ->join('psp_presupuesto', 'psp_id = dps_id_psp')
            ->join('pcl_proyecto_cliente', 'pcl_id = psp_id_pcl','left')
            ->join('tgp_tipo_gasto_presupuesto', 'tgp_id = dgp_id_tgp','left')
        ;
            
        if ($where != null) {
            $this->db->where($where);
        }
        
        if ($groupby != null) {
            $this->db->group_by($groupby,false);
        } else {
            $this->db->group_by('dgp_id_dps, YEAR(dgp_fecha), MONTH(dgp_fecha)',false);
        }
        
        if ($orderby != null) {
            $this->db->order_by($orderby);
        }
        
        
        $this->db->order_by('dgp_id_dps, MONTH(dgp_fecha)','DESC',false);
        
        return $query->get()->result_array();
    }
    
    /**
     * @author Luis Navarro - Grupo Satélite
     * @since 2018-12-21
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Obtener los presupuestos
     */
    public function obtener_presupuesto($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select('
                tpp_nombre,
                suc_nombre,
                fnc_nombre,
                (
                    SELECT CONCAT_WS(" ",emp_nombre, emp_apellido) FROM emp_empleado
                    WHERE emp_id = psp_id_emp_ejecutor
                ) AS ejecutor,
                (
                    SELECT CONCAT_WS(" ",emp_nombre, emp_apellido) FROM emp_empleado
                    WHERE emp_id = psp_id_emp_seguimiento
                ) AS seguimiento,
                are_nombre,
                cec_nombre,
                pcl_nombre,
                psp_presupuesto.*
            ', false)
            ->from('psp_presupuesto')
            ->join('tpp_tipo_presupuesto', 'tpp_id = psp_id_tpp', 'left')
            ->join('suc_sucursal', 'suc_id = psp_id_suc', 'left')
            ->join('fnc_financiador', 'fnc_id = psp_id_fnc', 'left')
            ->join('are_area', 'are_id = psp_id_are', 'left')
            ->join('cec_centro_costo', 'cec_id = psp_id_cec', 'left')
            ->join('pcl_proyecto_cliente', 'pcl_id = psp_id_pcl', 'left')
            ;
            
            if ($where != null) {
                $this->db->where($where);
            }
            
            if ($groupby != null) {
                $this->db->group_by($groupby);
            }
            
            if ($orderby != null) {
                $this->db->order_by($orderby);
            }
            
            return $query->get()->result_array();
    }


        /**
     * @author Francisco Orozco - Grupo Satélite
     * @since 2018-09-10
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Obtener los detalles de ejecucion de gastos por area
     */
    public function obtener_detalle_ejecucion_gastos_agrupado($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select("
                    ace_id,
                    dps_id,
                    ace_nombre,
                    sum(dps_presupuesto_actividad) as presupuesto,
                ", false)
        ->from('dps_detalle_presupuesto')
        ->join('psp_presupuesto', 'psp_id = dps_id_psp')
        ->join('ace_actividad_especifica', 'ace_id = dps_id_ace', 'left')
        ;
        
        if ($where != null) {
            $this->db->where($where);
        }
        
        if ($groupby != null) {
            $this->db->group_by($groupby);
        }
        
        if ($orderby != null) {
            $this->db->order_by($orderby);
        }
        
        return $query->get()->result_array();
    }

        /**
     * @author Luis Navarro - Grupo Satélite
     * @since 2018-01-08
     * @param $where = Condicionales para filtrar, $groupby = agrupar los datos, $orderby = ordenar por campos
     * @return Obtener los Gastos por Detalle de presupuesto 
     */
    public function reporte_detalle_gastos_ejecucion_area($where = null, $groupby = null, $orderby = null) {
        $query = $this->db->select("
                CONCAT_WS(
                    '_',
                    dps_id_ace,
                    DATE_FORMAT(dgp_fecha, '%m'),
                    DATE_FORMAT(dgp_fecha, '%Y')
                ) clave,
                MONTH (dgp_fecha) AS nmes,
                DATE_FORMAT(dgp_fecha, '%M') AS mes,
                DATE_FORMAT(dgp_fecha, '%Y') AS anio,
                IFNULL(SUM(dgp_monto), 0) AS suma
                ", false)
            ->from('dgp_detalle_gasto_prespuesto')
            ->join('dps_detalle_presupuesto', 'dgp_id_dps = dps_id')
            ;
            
            if ($where != null) {
                $this->db->where($where);
            }
            
            if ($groupby != null) {
                $this->db->group_by($groupby,false);
            } else {
                $this->db->group_by('dgp_id_dps, YEAR(dgp_fecha), MONTH(dgp_fecha)',false);
            }
            
            if ($orderby != null) {
                $this->db->order_by($orderby);
            }
            
            return $query->get()->result_array();
    }

    /**
     * @author Francisco Orozco - Grupo Satélite
     * @since 2019-02-25
     * @param $where = Condicionales para filtrar, $group_by = agrupar los datos
     * @return Obtener los detalles de gastos
    */
    public function get_detalles_presupuesto($where = null, $group_by = null) {
        $query = $this->db->select("*,
                IFNULL(SUM(dps_presupuesto_actividad),0) as presupuesto_actividad,
                (
                    SELECT
                        IFNULL(SUM(dgp_monto), 0)
                    FROM
                        dgp_detalle_gasto_prespuesto
                    WHERE
                        dgp_id_dps = dps_id
                        and  dgp_estado = 1
                ) AS monto_ejecutado,
                (
                    SELECT
                        IFNULL(
                            (
                                sum(orc_total) 
                            ) ,
                            0
                        )
                    FROM
                        orc_orden_compra
                    LEFT JOIN dpv_documento_proveedor ON dpv_id_orc = orc_id
                    WHERE
                        orc_id_psp = dps_id_psp
                    AND orc_id_ace = dps_id_ace
                    AND dpv_id IS NULL
                ) AS monto_comprometido",false)
            ->from('dps_detalle_presupuesto')
            ->join('psp_presupuesto', 'psp_id = dps_id_psp')
            ->join('ace_actividad_especifica', 'ace_id = dps_id_ace')                
            ->join('acg_actividad_general', 'acg_id = ace_id_acg')               
            ->join('pcl_proyecto_cliente', 'pcl_id = psp_id_pcl','left')
        ;

        if ($where != null) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-03-03
        * @param $where = Condicionales para filtrar, $group_by = agrupar los datos
        * @return Obtener los actividades generales
    */
    public function get_actividades_generales($where = null, $group_by = null) {
        $query = $this->db->select("*,
                (
                    SELECT
                        GROUP_CONCAT(uxa_id_usu)
                    FROM
                        uxa_usuarioxactividad_general
                    WHERE
                        uxa_id_acg = acg_id
                ) AS usuarios",false)
            ->from('acg_actividad_general')
        ;

        if ($where != null) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-03-03
        * @param $where = Condicionales para filtrar, $group_by = agrupar los datos
        * @return Obtener los actividades generales
    */
    public function get_planes_compras($where = null, $group_by = null) {
        $query = $this->db->select('*,
                DATE_FORMAT(dps_fecha_desde, "%m") mes_desde,
                DATE_FORMAT(dps_fecha_hasta, "%m") mes_hasta,
                DATE_FORMAT(dps_fecha_desde, "%Y") year_inicio,
                ( DATE_FORMAT(dps_fecha_hasta, "%Y") - DATE_FORMAT(dps_fecha_desde, "%Y")) years,
                (
                    SELECT
                        IFNULL(SUM(dgp_monto),0)
                    FROM
                        dgp_detalle_gasto_prespuesto
                    WHERE
                        dgp_id_dps = dps_id
                    AND dgp_estado = 1
                    AND DATE(dgp_fecha) BETWEEN DATE(pcp_fecha) and DATE(LAST_DAY(pcp_fecha))
                ) AS monto_ejecutado',false)
            ->from('dps_detalle_presupuesto')
            ->join('psp_presupuesto','psp_id = dps_id_psp')
            ->join('pcp_plan_compra_presupuesto','pcp_id_dps = dps_id','left')
            ->join('ace_actividad_especifica','ace_id = dps_id_ace')
        ;

        if ($where != null) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-03-03
        * @param $where = Condicionales para filtrar, $group_by = agrupar los datos
        * @return Obtener los actividades generales
    */
    public function get_plan_compras_mensual($where = null, $group_by = null, $order_by = null) {
        $query = $this->db->select('*,
                CONCAT_WS(
                    "_",
                    dps_id,
                    DATE_FORMAT(pcp_fecha, "%m"),
                    DATE_FORMAT(pcp_fecha, "%Y")
                ) clave,
                MONTH (pcp_fecha) AS nmes,
                DATE_FORMAT(pcp_fecha, "%M") AS mes,
                DATE_FORMAT(pcp_fecha, "%Y") AS anio,
                IFNULL(pcp_monto, 0) AS suma',false)
            ->from('pcp_plan_compra_presupuesto')
            ->join('dps_detalle_presupuesto','dps_id = pcp_id_dps')
            ->join('acg_actividad_general','acg_id = dps_id_acg')
            ->join('ace_actividad_especifica','ace_id = dps_id_ace')
        ;

        if ($where != null) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        if ($order_by != null) {
            $this->db->order_by($order_by);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francsico Orozco
        * @since 2019-06-10
        * @return artituculos
        * @param $where array asociativo para condicionar.
        * @param $like string para condicionar.
        * @param $where_in array.
    */
    public function get_articulos_like($where = null, $like = null, $where_in = null){
        $query = $this->db->select("art_id as id, art_nombre as nombre, art_codigo as codigo, art_estado as estado",false)
                ->from('art_articulo')
        ;

        if ($like != null) {
              $this->db->like("CONCAT_WS(' ', art_nombre, art_codigo)", $like);
              $this->db->limit(100);
        }

        if ($where != null) {
              $this->db->where($where);
        }

        if ($where_in != null) {
            $this->db->where_in($where_in);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Marlon Olmedo
        * @since 2019-06-27
        * @return Porcentaje de Ejecución Mensual
        * @param $where array asociativo para condicionar.
    */
    public function get_porcentaje_eje_mensual($where=null, $pcp_fecha=null, $dgp_fecha=null, $dpv_fecha=null, $orc_fecha=null){
        $query = $this->db->select("* ,
                    (SELECT sum(pcp_monto) FROM pcp_plan_compra_presupuesto WHERE pcp_id_dps = dps_id $pcp_fecha) as PAC,
                    (SELECT sum(dgp_monto) FROM dgp_detalle_gasto_prespuesto WHERE dgp_id_dps = dps_id $dgp_fecha) as ejecutado,
                    (SELECT GROUP_CONCAT(dpv_id_orc) FROM dpv_documento_proveedor WHERE dpv_id_ace = ace_id and dpv_id_acg = acg_id $dpv_fecha) as id_orc,
                    (SELECT SUM(orc_total) FROM orc_orden_compra WHERE 
                    IF(id_orc != null,orc_id NOT IN ( id_orc ),1=1)
                    and orc_id_ace = ace_id and orc_id_acg = acg_id $orc_fecha and orc_estado = 3) as comprometido", false)
            ->from('acg_actividad_general')
            ->join('ace_actividad_especifica','ace_id_acg = acg_id')
            ->join('dps_detalle_presupuesto','dps_id_acg = acg_id and dps_id_ace = ace_id')
            /*->join('pcp_plan_compra_presupuesto','pcp_id_dps = dps_id')*/
        ;

        if ($pcp_fecha != null) {
            $this->db->join('pcp_plan_compra_presupuesto','pcp_id_dps = dps_id');
        }

        if ($where != null) {
              $this->db->where($where);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francsico Orozco
        * @since 2019-08-10
        * @return artituculos
        * @param $where array asociativo para condicionar.
    */
    public function get_proyectos($where = null){
        $query = $this->db->select("*, (IFNULL(pcl_monto_solicitado,0) - IFNULL(sum(psp_total),0)) as calculo",false)
                ->from('pcl_proyecto_cliente')
                ->join('psp_presupuesto','psp_id_pcl = pcl_id','LEFT')
                ->join('epr_estado_proyecto','epr_id = pcl_id_epr','INNER')
                ->group_by("pcl_id")
                ->order_by("pcl_nombre","asc")
        ;

        if ($where != null) {
              $this->db->where($where);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-08-13
        * @param $where = Condicionales para filtrar
        * @return Obtener los actividades generales
    */
    public function get_actividades_generalesxpresupuesto($where = null){
        $query = $this->db->select("*",false)
                ->from('acg_actividad_general')
                ->join("dps_detalle_presupuesto","dps_id_acg = acg_id","INNER")
                ->group_by("acg_id")
                ->order_by("acg_nombre","asc")
        ;

        if ($where != null) {
              $this->db->where($where);
        }

        return $query->get()->result_array();
    }

     /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-08-16
        * @param $where = Condicionales para filtrar
        * @return Obtener los traslados de presupuesto
    */
    public function get_traslados($where = null){
        $query = $this->db->select('*, ace1.ace_nombre as origen, ace2.ace_nombre as destino')
                ->from('tmp_traslado_monto_plan')
                ->join("dps_detalle_presupuesto as dps1","dps1.dps_id = tmp_id_dps_origen","INNER")              
                ->join("ace_actividad_especifica as ace1","ace1.ace_id = dps1.dps_id_ace","INNER")

                ->join("psp_presupuesto","psp_id = dps_id_psp","INNER")                
                ->join("pcl_proyecto_cliente","pcl_id = psp_id_pcl","INNER")

                ->join("dps_detalle_presupuesto as dps2","dps2.dps_id = tmp_id_dps_destino","INNER")
                ->join("ace_actividad_especifica as ace2","ace2.ace_id = dps2.dps_id_ace","INNER")
                ->group_by("tmp_id")
                ->order_by("tmp_fecha","desc")
        ;

        if ($where != null) {
              $this->db->where($where,false);
        }

        return $query->get()->result_array();
    }

    /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-08-16
        * @param $where = Condicionales para filtrar
        * @return Obtener los origenes del traslados 
    */
    public function get_traslados_origen($where = null){
        $query = $this->db->select()
                ->from('ptr_plan_traslado_origen')
                ->join("pcp_plan_compra_presupuesto","pcp_id = ptr_id_pcp","INNER")              
                ->join("dps_detalle_presupuesto","dps_id = pcp_id_dps","INNER")                
                ->join("acg_actividad_general","acg_id = dps_id_acg","INNER")                   
                ->join("ace_actividad_especifica","ace_id = dps_id_ace","INNER")
        ;

        if ($where != null) {
              $this->db->where($where,false);
        }

        return $query->get()->result_array();
    }

     /**
        * @author Francisco Orozco - Grupo Satélite
        * @since 2019-08-16
        * @param $where = Condicionales para filtrar
        * @return Obtener los destino del traslados 
    */
    public function get_traslados_destino($where = null){
            $query = $this->db->select()
                    ->from('ptd_plan_traslado_destino')
                    ->join("pcp_plan_compra_presupuesto","pcp_id = ptd_id_pcp","INNER")              
                    ->join("dps_detalle_presupuesto","dps_id = pcp_id_dps","INNER")                
                    ->join("acg_actividad_general","acg_id = dps_id_acg","INNER")                   
                    ->join("ace_actividad_especifica","ace_id = dps_id_ace","INNER")
            ;
    
            if ($where != null) {
                  $this->db->where($where,false);
            }
    
            return $query->get()->result_array();
        }

    public function get_presupuestos_comp($where = null){
        $group_by = array('date_format(psp_fecha_desde, "%Y")','acg_id');
        $this->db->select('
                    acg_id as actividad,
                    acg_nombre as nombre,
                    date_format(psp_fecha_desde, "%Y") as anios,
                    psp_total as presupuesto,
                    (sum(dgp_monto)) as p_real
                    ',false)
                 ->from('psp_presupuesto')
                 ->join('dps_detalle_presupuesto','dps_id_psp = psp_id','INNER')
                 ->join('are_area','psp_id_are =  are_id','LEFT')
                 ->join('acg_actividad_general','acg_id = dps_id_acg','INNER')
                 ->join('dgp_detalle_gasto_prespuesto', 'dgp_id_dps = dps_id','LEFT')
                 ->group_by($group_by)
                 ;
                 if ($where != false) {
                    $this->db->where($where);
                 }
            return $this->db->get()->result_array();
    }

}