<?php

/**
 * Descripcion
 *  todos las consultas correspondientes al modulo de cuentas por cobrar
 *  Modelo de principal - Módulo Cuentas por cobrar
 * @author:     René Castaneda
 * @version:    2017/06/02
 * @since:      2017/06/02
 * @package:    Erp Satelite
 * =================================================================================
 */
class Cuentas_cobrar_model extends CI_Model
{

    function __construct()
    {

        parent::__construct();
    }


    /**
     * @author René Castaneda
     * @since 2017-06-05
     * @return Datos de los abonos realizados por el cliente
     * @param $where array asociativo para condicionar
     */
    public function get_abonos_cliente($where = null, $group_by = null)
    {
        $this->db->select('abc_id, abc_fecha, abc_monto, abc_comprobante, abc_estado, IF (cli_personeria = 1, CONCAT_WS(" ", cli_nombre, cli_apellido), cli_apellido) AS cliente,
                IFNULL((SELECT ban_nombre FROM ban_banco WHERE ban_id = dab_id_ban),"") AS banco,
                IFNULL((SELECT SUM(cmr_monto) FROM cmr_comprobante_retencion JOIN abf_abono_factura ON cmr_id_abf = abf_id WHERE abf_id_abc = abc_id), 0) AS abono_ret,
                IFNULL((SELECT GROUP_CONCAT(CONCAT(frp_nombre, IF(ban_nombre IS NULL, "", " - "), IFNULL(ban_nombre,"")) separator ", ") FROM frp_forma_pago 
                    INNER JOIN dab_detalle_abono ON dab_id_frp = frp_id 
                    LEFT JOIN ban_banco ON ban_id = dab_id_ban
                    WHERE dab_id_abc = abc_id
                    ), " - ") AS forma_pago,
                (
                    SELECT
                        GROUP_CONCAT(fac_numero SEPARATOR ", ")
                    FROM
                        fac_factura
                    INNER JOIN abf_abono_factura ON abf_id_fac = fac_id
                    WHERE
                        abf_id_abc = abc_id
                        AND fac_estado != 0
                ) AS facturas', false)
            ->from('abc_abono_cliente')
            ->join('cli_cliente', 'cli_id = abc_id_cli')
            ->join('dab_detalle_abono', 'abc_id = dab_id_abc', 'left');
        if ($where) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        return $this->db->get()->result_array();
    }

    /**
     * @author René Castaneda
     * @since 2017-06-09
     * @return Datos de los anticipos utilizados en un abono
     * @param $where array asociativo para condicionar
     */
    public function get_anticipo_abono($where = null)
    {
        $this->db->select()
            ->from('ana_anticipo_abono')
            ->join('anl_anticipo_cliente', 'anl_id = ana_id_anl');
        if ($where) {
            $this->db->where($where);
        }

        return $this->db->get()->result_array();
    }

    /**
     * @author René Castaneda
     * @since 2017-06-09
     * @return Datos de las retenciones facturas utilizadas en un abono (pago)
     * @param $where array asociativo para condicionar
     */
    public function get_facturas_abono_ret($where = null)
    {
        $this->db->select()
            ->from('abf_abono_factura')
            ->join('cmr_comprobante_retencion', 'cmr_id_abf = abf_id');
        if ($where) {
            $this->db->where($where);
        }

        return $this->db->get()->result_array();
    }

    /**
     * @author René Castaneda
     * @since 2017-06-13
     * @return Select de los cobradores
     * @param $where array asociativo para condicionar
     */
    public function select_cobrador($where = null)
    {
        $this->db->join('emp_empleado', 'emp_id = cob_id_emp', 'left');
        $select = $this->gsatelite->get_dropdown('cob_cobrador', '{cob_codigo} - {cob_nombre} {cob_apellido}', '', $where, null, '', 'cob_id', true);
        return $select;
    }

    /**
     * @author René Castaneda
     * @since 2017-06-13
     * @return Datos de las facturas y sus abonos, cobrador y vendedor
     * @param $where array asociativo para condicionar
     */
    public function get_cuentas_cobrar_detallado($where = null, $order_by = null)
    {
        $this->db->select("fac_id, cli_id, tpg_nombre, cli_codigo, GROUP_CONCAT(DISTINCT CONCAT_WS(' ',cob_nombre,cob_apellido) SEPARATOR ', ') cobrador,
                        IF(cli_personeria=1,CONCAT_WS(' ',cli_nombre, cli_apellido),cli_apellido) cliente, CONCAT_WS(' ',ven_nombre,ven_apellidos) vendedor,
                        tif_nombre, tif_abreviacion, fac_numero, fac_fecha_fact, fac_fecha_vencimiento,
                        IF(((fac_fecha_vencimiento IS NOT NULL) AND (fac_fecha_fact IS NOT NULL)),(DATEDIFF(fac_fecha_vencimiento,fac_fecha_fact)),'') plazo_pago,
                        IF(((fac_fecha_vencimiento IS NOT NULL)),(DATEDIFF(CURDATE(),fac_fecha_vencimiento)),'') dias_vencido,
                         (fac_total - fac_total_retencion) fac_total, fac_monto_abono, (fac_monto_pendiente - fac_total_retencion) fac_monto_pendiente", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('abf_abono_factura', 'abf_id_fac = fac_id', 'left')
            ->join('abc_abono_cliente', 'abc_id = abf_id_abc AND abc_estado = 1', 'left')
            ->join('cob_cobrador', 'cob_id = abc_id_cob', 'left')
            ->join('tpg_tipo_pago', 'tpg_id = fac_id_tpg', 'left')
            ->group_by('fac_id')
            ->having('fac_monto_pendiente > 0');
        if ($where) {
            $this->db->where($where);
        }
        if ($order_by) {
            foreach ($order_by as $key => $value) {
                $this->db->order_by($key, $value);
            }
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author Ronald Navarrete
     * @since 2023-02-24
     * @return Datos de remesas 
     */
    public function get_remesas_diarias_cxc($id_tipo_pago, $id_tipo_frp, $fecha_inicio, $fecha_final)
    {
        $query = "SELECT
                    CONCAT(
                        IFNULL( ( SELECT `sef_abrev` FROM `cof_correlativo_factura` JOIN `sef_serie_factura` ON `sef_id` = `cof_id_sef` WHERE `cof_id` = `fac_id_cof` AND `sef_id_tif` = `fac_id_tif` ), '' ),
                        ' - ',
                        `fac_numero` 
                    ) AS comprobante,
                    DATE_FORMAT( `fac_fecha_fact`, '%d-%m-%Y' ) AS fecha,
                    IF(cli_personeria = 2, cli_apellido,CONCAT_WS(' ',cli_nombre,cli_apellido)) AS cliente,
                    TRUNCATE (( `fac_descuento` + `fac_subtotal` ), 2 ) AS valor,
                    TRUNCATE ( `fac_descuento`, 1 ) AS descuento,
                    fac_subtotal,
                    fac_descuento,
                    fac_impuesto,
                    fac_retencion_abono,
                    fac_total_retencion,
                    abf_monto,
                    fac_total,
                    fac_id_tpg
                FROM
                    `abc_abono_cliente` abc,
                    `fac_factura` fac,
                    `cli_cliente` cli,
                    `abf_abono_factura` abf 
                WHERE
                    `fac_estado` > 0
                    AND`abf_id_abc` = `abc_id` 
                    AND `abf_id_fac` = `fac_id` 
                    AND `abc_id_cli` = `cli_id` 
                    AND `cli_id` = `fac_id_cli`";

        // TIPO DE PAGO
        if ($id_tipo_pago != '') {
            $query .= ' AND `fac_id_tpg` = ' . $id_tipo_pago;
        }

        // FORMA DE PAGO
        if ($id_tipo_frp != '') {
            $query .= ' AND `fac_id_frp` = ' . $id_tipo_frp;
        }

        // CONDICIONALES DE FECHA DESDE 
        if ($fecha_inicio != '') {
            $query .= ' AND date(`fac_fecha_fact`) >= "' . $fecha_inicio . '" ';
        }

        // CONDICIONALES DE FECHA HASTA
        if ($fecha_final != '') {
            $query .= ' AND date(`fac_fecha_fact`) <= "' . $fecha_final . '" ';
        }

        $query .= " GROUP BY
                        `fac_id`
                    HAVING
                        `fac_total` > 0 
                    ORDER BY 
                        `fac_id_tpg` ASC";

        $data = $this->db->query($query);

        return $data->result_array();
    }

    /**
     * @author Ronald Navarrete
     * @since 2023-02-24
     * @return Conteo de filas para saber donde termina la suma del tipo de pago y donde empieza el siguiente tipo
     */
    public function get_count_tipo_pago($id_tipo_pago, $id_tipo_frp, $fecha_inicio, $fecha_final)
    {
        if ($id_tipo_pago != '') {
            $query = "SELECT 
                        CONCAT(
                        IFNULL( ( SELECT `sef_abrev` FROM `cof_correlativo_factura` JOIN `sef_serie_factura` ON `sef_id` = `cof_id_sef` WHERE `cof_id` = `fac_id_cof` AND `sef_id_tif` = `fac_id_tif` ), '' ),
                        ' - ',
                        `fac_numero` 
                        ) AS comprobante,
                        DATE_FORMAT( `fac_fecha_fact`, '%d-%m-%Y' ) AS fecha,
                        IF(cli_personeria = 2, cli_apellido,CONCAT_WS(' ',cli_nombre,cli_apellido)) AS cliente,
                        TRUNCATE (( `fac_descuento` + `fac_subtotal` ), 2 ) AS valor,
                        TRUNCATE ( `fac_descuento`, 1 ) AS descuento,
                        fac_subtotal,
                        fac_descuento,
                        fac_impuesto,
                        fac_retencion_abono,
                        fac_total_retencion,
                        abf_monto,
                        fac_total,
                        fac_id_tpg
                    FROM
                        `abc_abono_cliente` abc,
                        `fac_factura` fac,
                        `cli_cliente` cli,
                        `abf_abono_factura` abf 
                    WHERE
                        `fac_estado` > 0
                        AND`abf_id_abc` = `abc_id` 
                        AND `abf_id_fac` = `fac_id` 
                        AND `abc_id_cli` = `cli_id` 
                        AND `cli_id` = `fac_id_cli`
                        AND `fac_id_tpg` = " . $id_tipo_pago . " ";

            // FORMA DE PAGO
            if ($id_tipo_frp != '') {
                $query .= ' AND `fac_id_frp` = ' . $id_tipo_frp . ' ';
            }

            // CONDICIONALES DE FECHA DESDE 
            if ($fecha_inicio != '') {
                $query .= ' AND date(`fac_fecha_fact`) >= "' . $fecha_inicio . '" ';
            }

            // CONDICIONALES DE FECHA HASTA
            if ($fecha_final != '') {
                $query .= ' AND date(`fac_fecha_fact`) <= "' . $fecha_final . '" ';
            }

            $query .= " GROUP BY
                            `fac_id`
                        HAVING
                            `fac_total` > 0 
                        ORDER BY 
                            `fac_id_tpg` ASC";
            $response = $this->db->query($query);
            $data[($id_tipo_pago == 2) ? 'Contado' : 'Credito'] = $response->num_rows();
        } else {
            for ($i = 0; $i <= 2; $i++) {
                $query = "SELECT 
                        CONCAT(
                        IFNULL( ( SELECT `sef_abrev` FROM `cof_correlativo_factura` JOIN `sef_serie_factura` ON `sef_id` = `cof_id_sef` WHERE `cof_id` = `fac_id_cof` AND `sef_id_tif` = `fac_id_tif` ), '' ),
                        ' - ',
                        `fac_numero` 
                        ) AS comprobante,
                        DATE_FORMAT( `fac_fecha_fact`, '%d-%m-%Y' ) AS fecha,
                        IF(cli_personeria = 2, cli_apellido,CONCAT_WS(' ',cli_nombre,cli_apellido)) AS cliente,
                        TRUNCATE (( `fac_descuento` + `fac_subtotal` ), 2 ) AS valor,
                        TRUNCATE ( `fac_descuento`, 1 ) AS descuento,
                        fac_subtotal,
                        fac_descuento,
                        fac_impuesto,
                        fac_retencion_abono,
                        abf_monto,
                        fac_total,
                        fac_id_tpg
                    FROM
                        `abc_abono_cliente` abc,
                        `fac_factura` fac,
                        `cli_cliente` cli,
                        `abf_abono_factura` abf 
                    WHERE
                        `fac_estado` > 0
                        AND`abf_id_abc` = `abc_id` 
                        AND `abf_id_fac` = `fac_id` 
                        AND `abc_id_cli` = `cli_id` 
                        AND `cli_id` = `fac_id_cli`
                        AND `fac_id_tpg` = " . $i . " ";

                // FORMA DE PAGO
                if ($id_tipo_frp != '') {
                    $query .= ' AND `fac_id_frg` = ' . $id_tipo_frp . ' ';
                }

                // CONDICIONALES DE FECHA DESDE 
                if ($fecha_inicio != '') {
                    $query .= ' AND date(`fac_fecha_fact`) >= "' . $fecha_inicio . '" ';
                }

                // CONDICIONALES DE FECHA HASTA
                if ($fecha_final != '') {
                    $query .= ' AND date(`fac_fecha_fact`) <= "' . $fecha_final . '" ';
                }

                $query .= " GROUP BY
                            `fac_id`
                        HAVING
                            `fac_total` > 0 
                        ORDER BY 
                            `fac_id_tpg` ASC";
                $response = $this->db->query($query);
                $data[($i == 2) ? 'Contado' : 'Credito'] = $response->num_rows();
            }
        }

        return $data;
    }


    public function get_seguimiento_cxcobrar($where = null)
    {
        $tipo_factura = json_decode($this->gsatelite->get_parametro('SYS_TIPO_FACTURAS'), true);
        $not_cred = (isset($tipo_factura['NOTA CREDITO'])) ? $tipo_factura['NOTA CREDITO'] : 0;
        $not_remi = (isset($tipo_factura['NOTA REMISION'])) ? $tipo_factura['NOTA REMISION'] : 0;
        $excluidas = array($not_cred, $not_remi);
        $this->db->select("fac_id, cli_id, cli_codigo, IF(cli_personeria = 1, CONCAT_WS(' ', cli_nombre, cli_apellido), cli_apellido) AS cliente, ven_codigo,
            CONCAT_WS(' ', ven_nombre, ven_apellidos) AS vendedor, GROUP_CONCAT(DISTINCT CONCAT_WS(' ', cob_codigo, cob_nombre, cob_apellido) SEPARATOR ', ') AS cobrador,
            tif_nombre, tif_abreviacion, fac_numero, fac_fecha_fact, fac_fecha_vencimiento, fac_fecha_prox_cobro, fac_fecha_seguimiento_cobro,fac_comentario_seguimiento,
            (SELECT sef_abrev FROM sef_serie_factura JOIN cof_correlativo_factura ON cof_id_sef = sef_id WHERE cof_id = fac_id_cof AND sef_id_tif = fac_id_tif) AS serie,
            IF(((fac_fecha_vencimiento IS NOT NULL) AND (fac_fecha_fact IS NOT NULL)), (DATEDIFF(fac_fecha_vencimiento, fac_fecha_fact)), '') AS plazo_pago,
            IF(((fac_fecha_vencimiento IS NOT NULL)), (DATEDIFF(CURDATE(), fac_fecha_vencimiento)), '') AS dias_vencido, fac_monto_abono, fac_estado, fac_total_retencion,
            (fac_total - fac_total_retencion) AS fac_total, fac_monto_pendiente, (fac_monto_pendiente - fac_total_retencion) saldo_pendiente", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('abf_abono_factura', 'abf_id_fac = fac_id', 'left')
            ->join('abc_abono_cliente', 'abc_id = abf_id_abc AND abc_estado = 1', 'left')
            ->join('cob_cobrador', 'cob_id = abc_id_cob', 'left')
            ->group_by('fac_id')
            ->order_by('cli_codigo')
            ->where_not_in('fac_id_tif', $excluidas);
        if ($where) {
            $this->db->where($where);
        }
        return $this->db->get()->result_array();
    }

    /**
     * @author René Castaneda
     * @since 2017-06-13
     * @return Datos de la factura
     * @param $where array asociativo para condicionar
     */
    public function get_datos_factura($where = null)
    {
        $this->db->select()
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('tpg_tipo_pago', 'tpg_id = fac_id_tpg', 'left')
            ->group_by('fac_id');
        if ($where) {
            $this->db->where($where);
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author René Castaneda
     * @since 2017-06-13
     * @return Detalles de la factura
     * @param $where array asociativo para condicionar
     */
    public function get_detalles_factura($where = null)
    {
        $this->db->select()
            ->from('def_detalle_fact')
            ->join('art_articulo', 'art_id = def_id_art', 'left')
            ->join('cva_conversion_articulo', 'cva_id = def_id_cva', 'left')
            ->join('mep_medida_producto', 'mep_id = cva_id_mep', 'left')
            ->group_by('def_id');
        if ($where) {
            $this->db->where($where);
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author René Castaneda
     * @since 2017-06-14
     * @return Datos de las facturas y sus saldos agrupados por cliente
     * @param $where array asociativo para condicionar
     */
    public function get_cuentas_cobrar_resumen($where = null)
    {
        $this->db->select("COUNT(fac_id) numero_facturas, cli_id, cli_codigo, IF(cli_personeria=1,CONCAT_WS(' ',cli_nombre, cli_apellido),cli_apellido) cliente, cli_codigo,
                        SUM(fac_total - fac_total_retencion) fac_total, SUM(fac_monto_abono) fac_monto_abono,
                        SUM(fac_monto_pendiente - fac_total_retencion) fac_monto_pendiente, GROUP_CONCAT( DISTINCT CONCAT_WS(' ',ven_nombre,ven_apellidos)) vendedor", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->group_by('fac_id_cli')
            ->having('fac_monto_pendiente > 0');
        if ($where) {
            $this->db->where($where);
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author René Castaneda
     * @since 2017-07-05
     * @return Datos de las facturas y sus saldos agrupados por cliente condicionando por rango de dias especificos.
     * @param $where array asociativo para condicionar
     */
    public function get_cuentas_antiguedad_saldo($where = null, $having = null, $desde = "", $hasta = "")
    {
        $tipo_factura = json_decode($this->gsatelite->get_parametro('SYS_TIPO_FACTURAS'), true);
        $not_cred = (isset($tipo_factura['NOTA CREDITO'])) ? $tipo_factura['NOTA CREDITO'] : 0;
        $not_remi = (isset($tipo_factura['NOTA REMISION'])) ? $tipo_factura['NOTA REMISION'] : 0;
        $excluidas = array($not_cred, $not_remi);
        $fecha_sin_vencer   = "";
        $fecha_rango1       = "";
        $fecha_rango2       = "";
        $fecha_rango3       = "";
        $fecha_rango4       = "";
        $fecha_restantes    = "";
        if ($desde == "" && $hasta != "") {
            $fecha_sin_vencer   = " AND DATE(f1.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango1       = " AND DATE(f2.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango2       = " AND DATE(f3.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango3       = " AND DATE(f4.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango4       = " AND DATE(f5.fac_fecha_fact) <= '$hasta' ";
            $fecha_restantes    = " AND DATE(f7.fac_fecha_fact) <= '$hasta' ";
        }
        if ($desde != "" && $hasta != "") {
            $fecha_sin_vencer   = " AND DATE(f1.fac_fecha_fact) >= '$desde' AND  DATE(f1.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango1       = " AND DATE(f2.fac_fecha_fact) >= '$desde' AND  DATE(f2.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango2       = " AND DATE(f3.fac_fecha_fact) >= '$desde' AND  DATE(f3.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango3       = " AND DATE(f4.fac_fecha_fact) >= '$desde' AND  DATE(f4.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango4       = " AND DATE(f5.fac_fecha_fact) >= '$desde' AND  DATE(f5.fac_fecha_fact) <= '$hasta' ";
            $fecha_restantes    = " AND DATE(f7.fac_fecha_fact) >= '$desde' AND  DATE(f7.fac_fecha_fact) <= '$hasta' ";
        }
        $this->db->select("COUNT(fac_id) numero_facturas, cli_id, cli_codigo,
                    IF(cli_personeria=1, CONCAT_WS(' ', cli_nombre, cli_apellido), cli_apellido) cliente,
                    SUM((fac_total - fac_total_retencion)) AS fac_total, 
                    SUM(fac_monto_abono) AS fac_monto_abono, 
                    SUM((fac_monto_pendiente - fac_total_retencion)) AS fac_monto_pendiente,
                    GROUP_CONCAT( DISTINCT CONCAT_WS(' ', ven_nombre, ven_apellidos) ) vendedor, 
                    IFNULL((
                        SELECT SUM((f1.fac_monto_pendiente - f1.fac_total_retencion)) from fac_factura f1
                        WHERE f1.fac_id_cli = cli_id and f1.fac_estado = 1 and  DATEDIFF(CURDATE(),f1.fac_fecha_vencimiento)<=0 $fecha_sin_vencer
                        )
                    , 0) AS sin_vencer, 
                    IFNULL((
                        SELECT SUM((f2.fac_monto_pendiente - f2.fac_total_retencion)) from fac_factura f2
                        WHERE f2.fac_id_cli = cli_id and f2.fac_estado = 1 and DATEDIFF(CURDATE(),f2.fac_fecha_vencimiento) BETWEEN  1 AND 30 $fecha_rango1
                        )
                    , 0) AS rango1,
                    IFNULL((
                        SELECT SUM((f3.fac_monto_pendiente - f3.fac_total_retencion)) from fac_factura f3
                        WHERE f3.fac_id_cli = cli_id and f3.fac_estado = 1 and  DATEDIFF(CURDATE(),f3.fac_fecha_vencimiento) BETWEEN  31 AND 60 $fecha_rango2
                        )
                    , 0) AS rango2, 
                    IFNULL((
                        SELECT SUM((f4.fac_monto_pendiente - f4.fac_total_retencion)) from fac_factura f4
                        WHERE f4.fac_id_cli = cli_id and f4.fac_estado = 1 and DATEDIFF(CURDATE(),f4.fac_fecha_vencimiento) BETWEEN  61 AND 90 $fecha_rango3
                        )
                    , 0) AS rango3, 
                    IFNULL((
                        SELECT SUM((f5.fac_monto_pendiente - f5.fac_total_retencion)) from fac_factura f5
                        WHERE f5.fac_id_cli = cli_id and f5.fac_estado = 1 and DATEDIFF(CURDATE(),f5.fac_fecha_vencimiento) BETWEEN  91 AND 120 $fecha_rango4
                        )
                    , 0) AS rango4,
                    IFNULL((
                        SELECT SUM((f7.fac_monto_pendiente - f7.fac_total_retencion)) from fac_factura f7
                        WHERE f7.fac_id_cli = cli_id and f7.fac_estado = 1 and DATEDIFF(CURDATE(),f7.fac_fecha_vencimiento) > 120 $fecha_restantes
                        ) 
                    , 0) AS restantes,
                     scl_codigo, scl_nombre
                     ", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('scl_sucursal_cliente', 'scl_id = fac_id_scl', 'left')
            ->where_not_in('fac_id_tif', $excluidas)
            ->group_by('fac_id_cli');
        if ($where) {
            $this->db->where($where);
        }

        if ($having) {
            $this->db->having($having, false);
        }

        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author René Castaneda
     * @since 2017-07-05
     * @return Datos de las facturas y sus saldos agrupados por cliente y mostrando la dirección
     * @param $where array asociativo para condicionar
     */
    public function get_cuentas_cobrar_direcciones($where = null)
    {
        $this->db->select("COUNT(fac_id) AS numero_facturas, cli_codigo, cli_id, IF(cli_fecha_cobro IS NULL,'',DATE_FORMAT(cli_fecha_cobro,'%d-%m-%Y')) AS cli_fecha_cobro,
                        IF(cli_personeria=1,CONCAT_WS(' ',cli_nombre, cli_apellido),cli_apellido) AS cliente, SUM((fac_total - fac_total_retencion)) AS fac_total, 
                        SUM(fac_monto_abono) AS fac_monto_abono, SUM((fac_monto_pendiente - fac_total_retencion)) AS fac_monto_pendiente, 
                        GROUP_CONCAT(DISTINCT CONCAT_WS(' ',ven_nombre,ven_apellidos)) AS vendedor,
                        (SELECT GROUP_CONCAT(DISTINCT CONCAT_WS(' ', cob_nombre, cob_apellido) SEPARATOR ', ') FROM abf_abono_factura JOIN abc_abono_cliente ON abc_id = abf_id_abc
                            JOIN cob_cobrador ON cob_id = abc_id_cob LEFT JOIN emp_empleado ON emp_id = cob_id_emp WHERE abc_estado = 1 AND abf_id_fac = fac_id) AS cobrador,
                        GROUP_CONCAT(DISTINCT cli_direccion) AS direccion, GROUP_CONCAT(DISTINCT CONCAT(tif_abreviacion,'-',fac_numero)) AS facturas, 
                        GROUP_CONCAT(DISTINCT mun_nombre) AS municipio, GROUP_CONCAT(DISTINCT txc_numero) telefono, GROUP_CONCAT(DISTINCT dep_nombre) AS departamento, 
                        IF(((fac_fecha_vencimiento IS NOT NULL)),(DATEDIFF(CURDATE(),fac_fecha_vencimiento)),'') AS dias_vencido
                        ", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('dic_direccion_cliente', 'dic_id_cli = cli_id', 'left')
            ->join('mun_municipio', 'mun_id = cli_id_mun', 'left')
            ->join('dep_departamento', 'dep_id = mun_id_dep', 'left')
            ->join('txc_telefonoxcliente', 'cli_id = txc_id_cli', 'left')
            ->group_by('fac_id_cli')
            ->order_by('cli_fecha_cobro')
            ->having('fac_monto_pendiente > 0');
        if ($where) {
            $this->db->where($where);
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author René Castaneda
     * @since 2017-07-13
     * @return Datos de las facturas y estado de cuenta del cliente
     * @param $where array asociativo para condicionar
     */
    public function get_estado_cuenta_cliente($where = null, $order_by = null, $fecha_inicio = null, $flag_masiva = "", $hasta = "")
    {
        $tif_nota_credito   = $this->gsatelite->get_parametro('SYS_ID_TIPO_FACTURA_NOTA_CREDITO', 0);
        $nota_credito_str   = ($tif_nota_credito > 0) ? " AND fac_id_tif != $tif_nota_credito " : "";
        $abonos_hasta = "";
        if ($hasta != "") {
            $abonos_hasta = ",  (
                            SELECT
                                SUM(abf.abf_monto)
                            FROM
                                abf_abono_factura AS abf JOIN abc_abono_cliente AS abc ON abf.abf_id_abc = abc.abc_id
                            WHERE
                                abf.abf_id_fac = fac_id
                                AND date_format(abc.abc_fecha, '%Y-%m-%d') <= '" . $hasta . "'
                                AND abc.abc_estado = 1
                            ) AS abonos_hasta";
        }
        $this->db->select("fac_id, tpg_nombre,fac_estado, fac_fecha_vencimiento,
                    IF(fac_estado = 1, DATEDIFF(NOW(),fac_fecha_vencimiento), 0) AS dias_vencidos,
                    tpc_nombre, cli_id, cli_codigo, 
                    IF(cli_personeria=1,CONCAT_WS(' ',cli_nombre, cli_apellido),cli_apellido) cliente, 
                    CONCAT_WS(' ',ven_nombre,ven_apellidos) vendedor, 
                    GROUP_CONCAT( DISTINCT CONCAT_WS(' ',cob_nombre,cob_apellido) ) cobrador, 
                    tif_nombre, tif_abreviacion, fac_numero, fac_fecha_fact,
                    IF(((fac_fecha_vencimiento IS NOT NULL) AND (fac_fecha_fact IS NOT NULL)),(DATEDIFF(fac_fecha_vencimiento,fac_fecha_fact)),'') plazo_pago, 
                    IF(((fac_fecha_vencimiento IS NOT NULL)),(DATEDIFF(CURDATE(),fac_fecha_vencimiento)),'') dias_vencido,
                    fac_total, fac_total_retencion, fac_monto_abono, fac_monto_pendiente, scl_codigo, scl_nombre, 
                    IFNULL((SELECT SUM(IF(f2.fac_monto_pendiente > 0,(f2.fac_monto_pendiente - f2.fac_total_retencion),f2.fac_monto_pendiente)) FROM fac_factura f2 where f2.fac_id_cli = cli_id
                    AND f2.fac_estado = 1  AND date_format(f2.fac_fecha_fact, '%Y-%m-%d') < '$fecha_inicio' $nota_credito_str $flag_masiva),0) saldo_anterior " . $abonos_hasta, false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('tpc_tipo_cliente', 'tpc_id = cli_id_tpc', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('abf_abono_factura', 'abf_id_fac = fac_id', 'left')
            ->join('abc_abono_cliente', 'abc_id = abf_id_abc and abc_estado = 1', 'left')
            ->join('cob_cobrador', 'cob_id = abc_id_cob', 'left')
            // ->join('emp_empleado','emp_id = cob_id_emp', 'left')
            ->join('tpg_tipo_pago', 'tpg_id = fac_id_tpg', 'left')
            ->join('scl_sucursal_cliente', 'scl_id_cli = fac_id_cli', 'left')
            ->group_by('fac_id');
        if ($where) {
            $this->db->where($where);
        }
        if ($order_by) {
            foreach ($order_by as $key => $value) {
                $this->db->order_by($key, $value);
            }
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author Habby Lovo
     * @since 2017-08-23
     * @return Categorias de los productos
     * @param $where array asociativo para condicionar
     */
    public function get_categoria_like($categoria, $where = null)
    {
        $this->db->select();
        $this->db->from('tia_tipo_articulo');
        $this->db->like('tia_nombre', $categoria);
        $this->db->or_like('tia_codigo', $categoria);
        $this->db->limit(100);

        if ($where != null) {
            $this->db->where($where);
        }

        $this->db->where('tia_estado', 1);
        $query = $this->db->get();

        return $query->result_array();
    }

    public function get_prorrateo($where = null)
    {
        /*
            SELECT
                abf_id,
                CONCAT(emp_codigo,' - ',emp_nombre,' ',emp_apellido) as empleado,
                art_nombre,
                tia_nombre,
                (abf_monto + abf_monto_retencion) as monto_total,
                (def_subtotal + def_impuesto) as total_art,
                SUM(
                    ROUND((ROUND(((def_subtotal + def_impuesto)/(SELECT
                    SUM((def_subtotal + def_impuesto))
                    FROM
                    def_detalle_fact
                    WHERE def_id_fac = abf_id_fac)),2) * (abf_monto + abf_monto_retencion)),2)) as prorrateo
            FROM
                abf_abono_factura
            INNER JOIN abc_abono_cliente ON abc_id = abf_id_abc
            LEFT JOIN cob_cobrador ON cob_id = abc_id_cob
            LEFT JOIN emp_empleado ON emp_id = cob_id_emp
            INNER JOIN def_detalle_fact ON def_id_fac = abf_id_fac
            INNER JOIN art_articulo ON art_id = def_id_art
            INNER JOIN tia_tipo_articulo ON tia_id = art_id_tia
            WHERE abc_fecha > '2017-08-28 12:32:40'
            GROUP BY cob_id, art_id_tia
        */
        $this->db->select("abf_id,
                abc_comprobante,
                abc_fecha,
                CONCAT(cob_codigo,' - ',cob_nombre,' ',cob_apellido) as empleado,
                CONCAT(cli_codigo,' - ',IF(cli_personeria = 1,CONCAT(cli_nombre,' ',cli_apellido),cli_apellido)) AS cliente,
                CONCAT(tif_abreviacion,' #',fac_numero) AS factura,
                art_nombre,
                tia_nombre,
                SUM(
                    ROUND((ROUND(((def_subtotal + def_impuesto)/(SELECT
                    SUM((def_subtotal + def_impuesto))
                    FROM
                    def_detalle_fact
                    WHERE def_id_fac = abf_id_fac)),2) * (abf_monto + abf_monto_retencion)),2)) as prorrateo", false);
        $this->db->from('abf_abono_factura');
        $this->db->join('abc_abono_cliente', 'abc_id = abf_id_abc', 'inner');
        $this->db->join('cli_cliente', 'cli_id = abc_id_cli', 'inner');
        $this->db->join('cob_cobrador', 'cob_id = abc_id_cob', 'left');
        // $this->db->join('emp_empleado','emp_id = cob_id_emp','left');
        $this->db->join('def_detalle_fact', 'def_id_fac = abf_id_fac', 'inner');
        $this->db->join('fac_factura', 'fac_id = def_id_fac', 'inner');
        $this->db->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'inner');
        $this->db->join('art_articulo', 'art_id = def_id_art', 'inner');
        $this->db->join('tia_tipo_articulo', 'tia_id = art_id_tia', 'inner');
        $this->db->group_by('cob_id, art_id_tia');
        $this->db->order_by('cob_id', 'DESC');
        $this->db->having('prorrateo > 0');
        $this->db->order_by('abc_comprobante', 'ASC');
        if ($where != null) {
            $this->db->where($where);
        }
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * @author Francsico Orozco
     * @since 2018-01-02
     * @return clientes
     * @param $where array asociativo para condicionar.
     * @param $like string para condicionar.
     */
    public function get_clientes_like($where = null, $like = null)
    {
        $query = $this->db->select("*, IF (
                    cli_personeria = 1,
                    CONCAT_WS(' ', cli_nombre, cli_apellido),
                    cli_apellido
                ) AS nombre", false)
            ->from('cli_cliente');
        $this->db->limit(100);

        if ($like != null) {
            $this->db->like("CONCAT_WS(' ', cli_codigo, cli_nombre, cli_apellido)", $like);
        }
        if ($where != null) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }

    /**
     * @author Nelson Soriano
     * @since 2019-01-14
     * @return Vendedores
     * @param $where array asociativo para condicionar.
     * @param $like string para condicionar.
     */
    public function get_vendedores_like($where = null, $like = null)
    {
        $query = $this->db->select('*', false)
            ->from('ven_vendedor');
        $this->db->limit(100);

        if ($like != null) {
            $this->db->like("CONCAT_WS(' ', ven_codigo, ven_nombre, ven_apellidos)", $like);
        }
        if ($where != null) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }




    /**
     * @author Francsico Orozco
     * @since 2018-05-07
     * @return antiguedad_saldo
     * @param $where array asociativo para condicionar.
     * @modify Nelson Soriano 2019-01-14
     * Descripcion: se agrego filtro de cliente
     * @modify Raul Escalante 2019-02-18
     * Descripcion: se agrego filtro de sucursal
     */
    public function antiguedad_saldo_cliente($cliente = null, $vendedor = null, $sucursal = null, $estado_cliente = null, $fecha_inicio = null, $fecha_fin = null, $slc_tipo_pago = null)
    {
        $tipo_factura = json_decode($this->gsatelite->get_parametro('SYS_TIPO_FACTURAS'), true);
        $not_cred = (isset($tipo_factura['NOTA CREDITO'])) ? $tipo_factura['NOTA CREDITO'] : 0;
        $not_remi = (isset($tipo_factura['NOTA REMISION'])) ? $tipo_factura['NOTA REMISION'] : 0;
        $periodo_anterior   = date("Y-m-d", strtotime("-1 day", strtotime($fecha_inicio)));
        $estado_cliente_str = (is_null($estado_cliente)) ? " " : " AND cli_estado = " . $estado_cliente . " ";
        $cliente_str        = ($cliente != null) ? " AND cli_id = " . $cliente . " " : " ";
        $vendedor           = ($vendedor != null) ? " AND ven_id = " . $vendedor . " " : " ";
        $sucursal           = ($sucursal != null) ? " AND scl_id = " . $sucursal . " " : " ";
        $slc_tipo_pago_str  = ($slc_tipo_pago != null) ? " AND fac_id_tpg = " . $slc_tipo_pago . " " : " ";
        $tif_nota_credito   = $this->gsatelite->get_parametro('SYS_ID_TIPO_FACTURA_NOTA_CREDITO', 0);
        // $nota_credito_str   = ($tif_nota_credito > 0) ? " AND fac_id_tif != $tif_nota_credito " : "";
        $nota_credito_str   = ($not_cred > 0 || $not_remi > 0) ? " AND fac_id_tif NOT IN ( $not_cred, $not_remi ) " : "";

        $query = "
        SELECT * FROM 
        (
            (
                SELECT 
                    1                   AS 'mostrar',
                    cli_id              AS 'id',
                    cli_codigo          AS 'Codigo',
                    IF (cli_personeria = 1, CONCAT(cli_nombre,' ',cli_apellido),cli_apellido)   AS 'Nombre',
                    '$periodo_anterior' AS 'Fecha Trans.',
                    ''                  AS 'Vencimiento', 
                    ''                  AS 'Tipo',
                    ''                  AS 'Documento',
                    ''                  AS 'Debito',
                    ''                  AS 'Credito',
                    0                   AS 'Balance incial',
                    ''                  AS 'Vendedor'
                FROM    cli_cliente
                WHERE   cli_id NOT IN (
                    SELECT fac_id_cli FROM fac_factura WHERE DATE(fac_fecha_fact) < '$fecha_inicio'
                )
                $estado_cliente_str
                $cliente_str
                GROUP BY cli_id
            )
            UNION
            (
                SELECT 
                    1                   AS 'mostrar',
                    cli_id              AS 'id',
                    cli_codigo          AS 'Codigo',
                    IF (cli_personeria = 1, CONCAT(cli_nombre,' ',cli_apellido),cli_apellido)   AS 'Nombre',
                    '$periodo_anterior' AS 'Fecha Trans.',
                    DATE(fac_fecha_vencimiento) AS 'Vencimiento',
                    ''                  AS 'Tipo',
                    ''                  AS 'Documento',
                    ''                  AS 'Debito',
                    ''                  AS 'Credito',
                    (SUM((fac_total - fac_total_retencion)) - IFNULL((SELECT SUM(abf_monto) FROM abf_abono_factura JOIN abc_abono_cliente ON abc_id = abf_id_abc 
                        WHERE DATE(abc_fecha) < '$fecha_inicio' AND abf_id_fac IN (SELECT fac_id FROM fac_factura WHERE fac_id_cli = cli_id 
                            AND (DATE(fac_fecha_anulacion) < '$fecha_fin' OR fac_fecha_anulacion IS NULL) 
                    $nota_credito_str)), 0))    AS 'Balance incial',
                    CONCAT(ven_nombre,' ',ven_apellidos)            AS 'Vendedor'
                FROM fac_factura
                LEFT JOIN   ven_vendedor            ON ven_id = fac_id_ven 
                LEFT JOIN   scl_sucursal_cliente    ON scl_id = fac_id_scl
                INNER JOIN  cli_cliente             ON cli_id = fac_id_cli
                WHERE DATE(fac_fecha_fact) < '$fecha_inicio' 
                AND (DATE(fac_fecha_anulacion) < '$fecha_fin' OR fac_fecha_anulacion IS NULL) 
                $nota_credito_str
                $estado_cliente_str
                $cliente_str
                $sucursal
                $vendedor
                GROUP BY fac_id_cli
            )
            UNION
            (
                SELECT 
                    (IF((fac_total < 0 && fac_id_fac_credito IS NOT NULL), 0, 1)) AS 'mostrar',
                    cli_id                  AS 'id',
                    cli_codigo              AS 'Codigo',
                    IF (cli_personeria = 1, CONCAT(cli_nombre,' ',cli_apellido),cli_apellido)   AS 'Nombre',
                    DATE(fac_fecha_fact)    AS 'Fecha Trans.',
                    DATE(fac_fecha_vencimiento) AS 'Vencimiento',
                    tif_abreviacion         AS 'Tipo',
                    fac_numero              AS 'Documento',
                    (fac_total - fac_total_retencion)   AS 'Debito',
                    0                       AS 'Credito',
                    ''                      AS 'Balance incial',
                    CONCAT(ven_nombre, ' ', ven_apellidos)  AS 'Vendedor'
                FROM fac_factura
                JOIN        cli_cliente             ON cli_id = fac_id_cli
                JOIN        tif_tipo_factura        ON tif_id = fac_id_tif
                LEFT JOIN   ven_vendedor            ON ven_id = fac_id_ven
                LEFT JOIN   scl_sucursal_cliente    ON scl_id = fac_id_scl
                WHERE DATE(fac_fecha_fact) >= '$fecha_inicio'
                AND DATE(fac_fecha_fact) <= '$fecha_fin' 
                AND (DATE(fac_fecha_anulacion) <= '$fecha_fin' OR DATE(fac_fecha_anulacion) IS NULL)
                $nota_credito_str
                $estado_cliente_str
                $cliente_str
                $slc_tipo_pago_str
                $sucursal
                $vendedor
            )
            UNION
            (
                SELECT 
                    1                       AS 'mostrar',
                    cli_id                  AS 'id',
                    cli_codigo              AS 'Codigo',
                    IF (cli_personeria = 1, CONCAT(cli_nombre,' ',cli_apellido),cli_apellido)   AS 'Nombre',
                    DATE(abc_fecha)         AS 'Fecha Trans.',
                    DATE(fac_fecha_vencimiento) AS 'Vencimiento',
                    IFNULL(
                        (SELECT tif_abreviacion FROM fac_factura JOIN tif_tipo_factura ON fac_id_tif = tif_id WHERE fac_id_fac_credito = abf_id_fac AND abf_monto_entregado = 0 LIMIT 1),
                        'REC')              AS 'Tipo',
                    IFNULL(
                        (SELECT fac_numero from fac_factura WHERE fac_id_fac_credito = abf_id_fac and abf_monto_entregado = 0 LIMIT 1),
                        IFNULL(abc_comprobante, 
                            fac_numero
                            )
                        )                   AS 'Documento',
                    0                       AS 'Debito',
                    abf_monto               AS 'Credito',
                    ''                      AS 'Balance incial',
                    CONCAT(ven_nombre,' ',ven_apellidos)    AS 'Vendedor'
                FROM abf_abono_factura
                JOIN        fac_factura             ON fac_id = abf_id_fac
                JOIN        cli_cliente             ON cli_id = fac_id_cli
                JOIN        abc_abono_cliente       ON abc_id = abf_id_abc
                LEFT JOIN   scl_sucursal_cliente    ON scl_id = fac_id_scl
                LEFT JOIN   ven_vendedor            ON ven_id = fac_id_ven
                WHERE DATE(abc_fecha) >= '$fecha_inicio' AND DATE(abc_fecha) <= '$fecha_fin'
                AND abc_estado != 0
                $estado_cliente_str
                $cliente_str
                $sucursal
                $vendedor
                AND abf_id_fac IN (
                    SELECT  fac_id
                    FROM    fac_factura
                    INNER JOIN cli_cliente ON cli_id = fac_id_cli
                    WHERE   DATE(fac_fecha_fact) <= '$fecha_fin'
                    AND (DATE(fac_fecha_anulacion) <= '$fecha_fin' OR DATE(fac_fecha_anulacion) IS NULL)
                    $slc_tipo_pago_str 
                    $vendedor 
                    GROUP BY    fac_id_cli
                )
            )
            ORDER BY 'Codigo', 'Fecha Trans.' ASC
        ) AS a
        HAVING a.`mostrar` > 0
        ORDER BY
            a.`Codigo`,
            a.`Fecha Trans.`,
            a.`Documento`,
            a.`Tipo`,
            a.`Credito` ASC";

        $query = $this->db->query($query);
        return $query->result_array();
    }


    /**
     * @author Elman Ortiz
     * @since 04/07/2018
     * @method Se obtiene la Cuenta contable para el debe al hacer pago de una factura
     */
    public function get_pago_cliente_debe($abc_id, $tcn_id_debe)
    {
        // Cuenta contable por defecto
        $defecto_cnt = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt[0]['cnt_id'];
        $defecto_cnt_nombre = $this->gsatelite->get_campo('cnt_cuenta_contable', 'cnt_nombre', array('cnt_id' => $defecto_cnt));
        $tcn_id_debe_cub = $this->gsatelite->get_parametro('SYS_TIPO_CUENTAS', null, null, array('BANCOS'));
        /*****************
         * El subselect donde se obtiene el nombre y id de la cuenta contable vinculada al cliente 
         * retorna 2 resultado porque en la tabla cxn_cuentaxcliente existen dos registros del mismo cliente
         * con el mismo tipo de cuenta y la misma forma de pago.
         *********************/
        $query = "
            SELECT a.fac_id,
            SUM(a.total_abonar) AS total_abonar,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cnt_id,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cnt_nombre
            FROM
            (
            SELECT
                fac_id,
                abf_monto AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_debe . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_debe . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abc_id . "
            ) AS a
            GROUP BY cnt_id
        ";

        $datos_query = $this->db->query($query)->result_array();

        //Se verifica si el abono no tiene configuración a detalle de abonos por cuentas bancadias
        $query_detalle_abono = "
            SELECT 
                NULL as fac_id,
                dab_monto AS total_abonar,
                cnt_id,
                cnt_nombre
                FROM cnt_cuenta_contable 
                INNER JOIN cxb_cuentaxcuenta_bancaria ON cxb_id_cnt = cnt_id
                INNER JOIN tra_transaccion ON tra_id_cub = cxb_id_cub
                INNER JOIN dab_detalle_abono ON dab_id_tra = tra_id
                WHERE dab_id_abc = " . $abc_id . " AND cxb_id_tcn = " . $tcn_id_debe_cub . "
        ";

        $datos_detalle_query = $this->db->query($query_detalle_abono)->result_array();

        if (count($datos_detalle_query) > 0) {
            $suma_detalles = array_sum(array_column($datos_detalle_query, 'total_abonar'));
            /**
             * Si la suma de los detalles es igual a la suma del abono se retorna las cuentas detalle.
             * Sino, se resta el monto de los detalles, y se hace merge con el array
             */
            if ($datos_query[0]['total_abonar'] == $suma_detalles) {
                $datos_query = $datos_detalle_query;
            } else {
                $datos_query[0]['total_abonar'] -= $suma_detalles;
                $datos_query = array_merge($datos_query, $datos_detalle_query);
            }
        }

        return $datos_query;
    }

    /**
     * @author Raul E
     * @since 03/2023
     * @param $where array asociativo para condicionar.
     */
    public function get_comisiones_cxc($where = null)
    {
        $query = $this->db->select("fac_id, fac_numero, DATE(abc_fecha) AS fecha_pago, DATE(fac_fecha_fact) AS fecha_comp, abf_monto,
            (SELECT CONCAT(cob_codigo,' - ',cob_nombre,' ',cob_apellido) FROM cob_cobrador WHERE cob_id = abc_id_cob) AS cobrador,
            (SELECT CONCAT(ven_codigo,' - ',ven_nombre,' ',ven_apellidos) FROM ven_vendedor WHERE ven_id = fac_id_ven) AS vendedor,
            (SELECT sef_abrev FROM cof_correlativo_factura JOIN sef_serie_factura ON sef_id = cof_id_sef WHERE cof_id = fac_id_cof AND sef_id_tif = fac_id_tif) AS serie,
            (SELECT IF(cli_personeria = 1,CONCAT_WS(' ',cli_nombre,cli_apellido),cli_apellido) FROM cli_cliente WHERE cli_id = fac_id_cli) AS cliente,
            TIMESTAMPDIFF(DAY, DATE(fac_fecha_fact), DATE(abc_fecha)) AS dias, (fac_total - fac_total_retencion) AS total_fact,
            IFNULL(ROUND((SUM(def_subtotal) * (abf_monto / (fac_total - fac_total_retencion)) * (SELECT (crp_porcentaje / 100) FROM crp_comision_recuperacion 
                WHERE crp_estado = 1 AND crp_id_tia = art_id_tia AND crp_desde <= TIMESTAMPDIFF(DAY, DATE(fac_fecha_fact), DATE(abc_fecha)) 
                AND crp_hasta >= TIMESTAMPDIFF(DAY, DATE(fac_fecha_fact), DATE(abc_fecha)) ORDER BY crp_id DESC LIMIT 1)), 2), '') AS porc_com_estimado,
            (abf_monto / (fac_total - fac_total_retencion)) AS parte_pago, SUM(def_subtotal) AS suma_def,
            (SUM(def_subtotal) / (SELECT SUM(a.def_subtotal) FROM def_detalle_fact AS a WHERE a.def_id_fac = fac_id)) AS porcion_def,
            IFNULL( IF(ISNULL(abc_id_cob), CONCAT('V','&&', fac_id_ven), CONCAT('C','&&', abc_id_cob)), '') AS ven_cob,
            IFNULL( (SELECT IFNULL(CONCAT('1##',crp_valor), CONCAT('2##',crp_porcentaje)) FROM crp_comision_recuperacion WHERE crp_estado = 1 AND crp_id_tia = art_id_tia AND crp_desde <= TIMESTAMPDIFF(DAY, DATE(fac_fecha_fact), DATE(abc_fecha)) AND crp_hasta >= TIMESTAMPDIFF(DAY, DATE(fac_fecha_fact), DATE(abc_fecha)) ORDER BY crp_id DESC LIMIT 1 ), '') AS porcn_ctdad", false)
            ->from('def_detalle_fact')
            ->join('art_articulo', 'art_id = def_id_art')
            ->join('fac_factura', 'fac_id = def_id_fac')
            ->join('abf_abono_factura', 'abf_id_fac = fac_id')
            ->join('abc_abono_cliente', 'abc_id = abf_id_abc')
            ->having('porcn_ctdad <> "" AND ven_cob <> ""')
            ->group_by('def_id_fac, abf_id_abc, art_id_tia')
            ->order_by('ven_cob ASC, fac_fecha_fact ASC, dias ASC, abf_id_abc ASC');
        if ($where) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }

    /***************************************************************/
    /**
     * @author Elman Ortiz
     * @since 04/07/2018
     * @method get_pago_cliente_haber
     */
    public function get_pago_cliente_haber($abc_id, $tcn_id_haber)
    {
        $defecto_cnt = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt[0]['cnt_id'];
        $defecto_cnt_nombre = $this->gsatelite->get_campo('cnt_cuenta_contable', 'cnt_nombre', array('cnt_id' => $defecto_cnt));
        $query = "
            SELECT a.fac_id,
            SUM(a.total_abonar) AS total_abonar,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cnt_id,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cnt_nombre
            FROM
            (
            SELECT
                fac_id,
                abf_monto AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_haber . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_haber . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abc_id . "
               /* AND fac_id_tpg <> 1 */
            ) AS a
            GROUP BY cnt_id
        ";
        return $this->db->query($query)->result_array();
    }


    public function get_pagos_pendientes_partida($tipoCuentaDebe, $tipoCuentaHaber)
    {
        $query = $this->db->select()
            ->from('abc_abono_cliente')
            ->where('abc_id_pct IS NULL')
            //->limit(5)
        ;

        $abonosPendientes = $query->get()->result_array();

        //Se obtienen los datos de la cuenta por defecto
        $defecto_cnt = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt[0]['cnt_id'];
        $defecto_cnt_nombre = $this->gsatelite->get_campo('cnt_cuenta_contable', 'cnt_nombre', array('cnt_id' => $defecto_cnt));


        //Se obtienens las cuentas del DEBE
        foreach ($abonosPendientes as &$abono) {
            //Se obtiene los abonos de facturas por abono cliente

            $query = "
            SELECT
            a.total_abonar,
            a.fac_numero,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cuenta_debe,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cuenta_debe_nombre
            FROM
            (
            SELECT
                GROUP_CONCAT(fac_numero ORDER BY fac_numero ASC) as fac_numero,
                SUM(abf_monto) AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaDebe . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaDebe . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abono['abc_id'] . "
                GROUP BY abc_id
            ) AS a
        ";
            $abono['abono_factura_debe'] = $this->db->query($query)->result_array();
        }

        //Se obtienens las cuentas del HABER
        foreach ($abonosPendientes as &$abono) {
            //Se obtiene los abonos de facturas por abono cliente

            $query = "
            SELECT
            a.total_abonar,
            CASE WHEN fac_id_tpg = 2 AND cnt_haber_id_credito IS NOT NULL THEN cnt_haber_id_credito
            WHEN fac_id_tpg <> 2 AND cnt_haber_id_trans IS NOT NULL THEN cnt_haber_id_trans
            ELSE " . $defecto_cnt . "
            END AS cuenta_haber,
            CASE WHEN fac_id_tpg = 2 AND cnt_haber_id_credito IS NOT NULL THEN cnt_haber_credito_nombre
            WHEN fac_id_tpg <> 2 AND cnt_haber_id_trans IS NOT NULL THEN cnt_haber_trans_nombre
            ELSE '" . $defecto_cnt_nombre . "'
            END AS cuenta_haber_nombre,
            a.fac_numero
            FROM
            (
            SELECT
                fac_id_tpg,
                GROUP_CONCAT(fac_numero ORDER BY fac_numero ASC) as fac_numero,
                SUM(abf_monto) AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaHaber['tipoCuentaCredito'] . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_haber_id_credito,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaHaber['tipoCuentaCredito'] . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_haber_credito_nombre,
                 (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaHaber['tipoCuentaTransitoria'] . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_haber_id_trans,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tipoCuentaHaber['tipoCuentaTransitoria'] . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_haber_trans_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abono['abc_id'] . "
                GROUP BY abc_id
            ) AS a
        ";
            $abono['abono_factura_haber'] = $this->db->query($query)->result_array();
        }

        return $abonosPendientes;
    }

    /**
     * @author Mario Marquez
     * @since 29/08/2018
     * @method get_cuenta_contable_transaccion
     * Se obtiene la Cuenta contable para el debe al hacer pago de una factura
     */
    public function get_cuenta_contable_anticipo($anticipo_id, $tipo_cuenta)
    {
        // Cuenta contable por defecto
        $defecto_cnt = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt[0]['cnt_id'];
        $defecto_cnt_nombre = $this->gsatelite->get_campo('cnt_cuenta_contable', 'cnt_nombre', array('cnt_id' => $defecto_cnt));

        $query = "
            SELECT
            a.total_abonar,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cnt_id,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cnt_nombre
            FROM
            (
            SELECT
                anl.anl_monto AS total_abonar,
                (
                   SELECT
                        cnt_id
                    FROM cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                    cxn_id_tcn = " . $tipo_cuenta . "
                    AND cxn_id_tpg = 1
                    and cxn_id_cli = anl.anl_id_cli
                    LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                    cxn_id_tcn = " . $tipo_cuenta . "
                    AND cxn_id_tpg = 1
                    and cxn_id_cli = anl.anl_id_cli
                    LIMIT 1
                ) AS cnt_nombre
            FROM anl_anticipo_cliente anl
            WHERE anl.anl_id = " . $anticipo_id . "
            ) AS a
        ";
        return $this->db->query($query)->result_array();
    }

    public function get_rutas_cob($where = null)
    {
        $this->db->select("rtc_id,rtc_nombre")
            ->from('rxb_rutaxcobrador')
            ->join('rtc_ruta_cobro', 'rtc_id = rxb_id_rtc');
        if ($where != null) {
            $this->db->where($where);
        }
        return $this->db->get()->result_array();
    }

    public function get_clientes_ruta($where = null)
    {
        $query = $this->db->select("cli_codigo, IF(cli_personeria=2,cli_apellido,CONCAT_WS(' ',cli_nombre,cli_apellido)) as cliente, fac_total, fac_numero,
                    date_format(fac_fecha_fact,'%d-%m-%Y') as fecha, fac_total_retencion, fac_monto_pendiente, rtc_id, rtc_nombre", false)
            ->from('rxb_rutaxcobrador')
            ->join('rtc_ruta_cobro', 'rtc_id = rxb_id_rtc')
            ->join('cli_cliente', 'cli_id_rtc = rtc_id')
            ->join('fac_factura', 'fac_id_cli = cli_id');
        $this->db->having('(fac_monto_pendiente - fac_total_retencion) > 0');
        if ($where != null) {
            $this->db->where($where);
        }
        $this->db->order_by('rtc_id,cli_codigo');
        return $query->get()->result_array();
    }

    /**
     * @author Francisco Orozco
     * @since 2018-10-09
     * @return facturas afectadas por el abono.
     * @param $where array asociativo para condicionar
     */
    public function get_facturas_abonadas($where = null)
    {
        $this->db->select('*, IF (
                        cli_personeria = 1,
                        CONCAT(
                            cli_nombre,
                            " ",
                            cli_apellido
                        ),
                        cli_apellido
                    ) AS cliente,
                    (
                        SELECT
                            txc_numero
                        FROM
                            txc_telefonoxcliente
                        WHERE
                            txc_id_cli = cli_id
                        ORDER BY
                            txc_fecha_mod ASC
                        LIMIT 1
                    ) AS numero', false)
            ->from('abc_abono_cliente')
            ->join('cli_cliente', 'cli_id = abc_id_cli')
            ->join('abf_abono_factura', 'abf_id_abc = abc_id')
            ->join('fac_factura', 'fac_id = abf_id_fac')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif');
        if ($where) {
            $this->db->where($where);
        }
        return $this->db->get()->result_array();
    }

    /**
     * @author René Castaneda
     * @since 12/12/2018
     * @method Se obtiene la Cuenta contable para el debe al hacer pago de una factura transitoria
     */
    public function get_pago_cliente_debe_cxc($abc_id, $tcn_id_debe)
    {
        // Cuenta contable por defecto
        $defecto_cnt_reg = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt_reg[0]['cnt_id'];
        $defecto_cnt_nombre = $defecto_cnt_reg[0]['cnt_nombre'];

        $nfacturas_abonadas = $this->gsatelite->get_tabla('abf_abono_factura', array('abf_id_abc' => $abc_id));
        $nfacturas_abonadas = count($nfacturas_abonadas);

        $reg_anticipos_abono = $this->gsatelite->get_tabla('ana_anticipo_abono', array('ana_id_abc' => $abc_id));
        $total_anticipo = round(array_sum(array_column($reg_anticipos_abono, 'ana_monto')), 4);
        $resta_proporcional = round(($total_anticipo / $nfacturas_abonadas), 4);
        $resta_proporcional = ($resta_proporcional > 0) ? $resta_proporcional : 0;

        $query = "
            SELECT a.fac_id,
            SUM(a.total_abonar) AS total_abonar,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cnt_id,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cnt_nombre
            FROM
            (
            SELECT
                fac_id,
                 ROUND(( abf_monto - $resta_proporcional ),4)
                AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_debe . "
                    AND cxn_id_tpg = fac_id_tpg  LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_debe . "
                    AND cxn_id_tpg = fac_id_tpg LIMIT 1
                ) AS cnt_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abc_id . "
            ) AS a
            GROUP BY cnt_id
        ";
        return $this->db->query($query)->result_array();
    }

    /***************************************************************/
    /**
     * @author René Castaneda
     * @since 12/12/2018
     * @method get_pago_cliente_haber_cxc
     */
    public function get_pago_cliente_haber_cxc($abc_id, $tcn_id_haber)
    {
        $defecto_cnt_reg = $this->gsatelite->get_tabla('cnt_cuenta_contable', array('cnt_id' => $this->gsatelite->get_parametro('SYS_CUENTA_DEFECTO'), 'cnt_estado' => 1));
        $defecto_cnt = $defecto_cnt_reg[0]['cnt_id'];
        $defecto_cnt_nombre = $defecto_cnt_reg[0]['cnt_nombre'];

        $query = "
            SELECT a.fac_id,
            SUM(a.total_abonar) AS total_abonar,
            IF(ISNULL(a.cnt_id)," . $defecto_cnt . ",cnt_id) AS cnt_id,
            IF(ISNULL(a.cnt_id),'" . $defecto_cnt_nombre . "',a.cnt_nombre) AS cnt_nombre
            FROM
            (
            SELECT
                fac_id,
                (
                abf_monto
                - ( SELECT SUM(ixf_monto) FROM ixf_impuestoxfactura 
                    JOIN def_detalle_fact ON def_id = ixf_id_def
                WHERE def_id_fac = fac_id )
                ) 
                AS total_abonar,
                (
                    SELECT
                        cnt_id
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_haber . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_id,
                (
                    SELECT
                        cnt_nombre
                    FROM
                        cli_cliente
                    JOIN cxn_cuentaxcliente ON cli_id = cxn_id_cli
                    JOIN cnt_cuenta_contable ON cxn_id_cnt = cnt_id
                    WHERE
                        cli_id = fac_id_cli
                    AND cxn_id_tcn = " . $tcn_id_haber . "
                    AND cxn_id_tpg = fac_id_tpg
                    LIMIT 1
                ) AS cnt_nombre
            FROM
                fac_factura
            JOIN abf_abono_factura ON fac_id = abf_id_fac
            JOIN abc_abono_cliente ON abf_id_abc = abc_id
            WHERE
                abc_id = " . $abc_id . "
            ) AS a
            GROUP BY cnt_id
        ";
        return $this->db->query($query)->result_array();
    }

    /***************************************************************/
    /**
     * @author René Castaneda
     * @since 04/02/2019
     * @method get_facturas_anticipo 
     * @return array asociativo de la conincidencia encontrada sobre los anticipos de una factura.
     */
    public function get_facturas_anticipo($where)
    {
        $query = $this->db->select("*, COUNT(abc_id) n_abc, COUNT(ana_id) n_ana", false)
            ->from('abf_abono_factura')
            ->join('abc_abono_cliente', 'abc_id = abf_id_abc')
            ->join('ana_anticipo_abono', 'ana_id_abc = abc_id')
            ->group_by('abf_id_fac');
        if ($where) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }

    /**
     * @author Raul Escalante
     * @since 2019-2-18
     * @return sucursales
     * @param $where array asociativo para condicionar.
     * @param $like string para condicionar.
     */
    public function get_sucursal_like($where = null, $like = null)
    {
        $query = $this->db->select("scl_id,scl_codigo,scl_nombre", false)
            ->from('scl_sucursal_cliente');
        $this->db->limit(100);

        if ($like != null) {
            $this->db->like("CONCAT(scl_codigo,' - ',scl_nombre)", $like);
        }
        if ($where != null) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }

    /**
     * @author Raul Escalante
     * @since 2019-5-14
     * @return datos de los abonos realizados por clientes
     * @param $where array para condicionar.
     */
    public function get_clientes_lavado($where = null, $fecha_inicio = null, $fecha_fin = null)
    {
        $tpg = json_decode($this->gsatelite->get_parametro('SYS_ID_PAGO_EFECTIVO'), true);
        $cruces = ($tpg['cruce'] > 0) ? $tpg['cruce'] : 0;
        $cheque = ($tpg['cheque'] > 0) ? $tpg['cheque'] : 0;
        $efectv = ($tpg['efectivo'] > 0) ? $tpg['efectivo'] : 0;
        $tarjts = ($tpg['tarjetas'] > 0) ? $tpg['tarjetas'] : 0;
        $compns = ($tpg['compensacion'] > 0) ? $tpg['compensacion'] : 0;
        $transf = ($tpg['transferencias'] > 0) ? $tpg['transferencias'] : 0;
        $periodo = ($fecha_inicio && $fecha_fin) ? " AND DATE(abc_fecha) >= '" . $fecha_inicio . "' AND DATE(abc_fecha) <= '" . $fecha_fin . "' " : "";
        $sql = "c1.cli_id AS cli_id, c1.cli_codigo AS cli_codigo, abc_fecha,
                IF ( c1.cli_personeria = 1, CONCAT_WS(' ', c1.cli_nombre, c1.cli_apellido ), c1.cli_apellido ) AS cli_nombre,
                (SELECT SUM(abf_monto) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id
                    WHERE dab_id_frp = $cheque AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS cheque,
                (SELECT SUM( abf_monto ) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id
                    WHERE dab_id_frp = $transf AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS transferencia,
                (SELECT SUM( abf_monto ) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id
                    WHERE dab_id_frp = $efectv AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS efectivo,
                (SELECT SUM( abf_monto ) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id
                    WHERE dab_id_frp = $tarjts AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS tarjetas,
                (SELECT SUM( abf_monto ) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id
                    WHERE dab_id_frp IN ( $compns , $cruces ) AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS otros,
                (SELECT SUM(abf_monto) FROM fac_factura JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id 
                    JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id JOIN abc_abono_cliente ON abc_id = abf_id_abc LEFT JOIN dab_detalle_abono ON dab_id_abc = abc_id 
                    WHERE dab_id_frp IS NULL AND c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS sin_detalle,
                (SELECT GROUP_CONCAT(DISTINCT CONCAT(tif_abreviacion,'-',fac_numero) SEPARATOR ', ') FROM fac_factura JOIN tif_tipo_factura ON tif_id = fac_id_tif 
                    JOIN frp_forma_pago ON fac_id_frp = frp_id JOIN abf_abono_factura ON abf_id_fac = fac_id JOIN cli_cliente AS c2 ON fac_id_cli = c2.cli_id 
                    JOIN abc_abono_cliente ON abc_id = abf_id_abc JOIN dab_detalle_abono ON dab_id_abc = abc_id WHERE c2.cli_id = c1.cli_id AND abc_estado = 1 $periodo) AS facturas
                ";
        $query = $this->db->select($sql, false)
            ->from('abc_abono_cliente')
            ->join('cli_cliente AS c1', 'c1.cli_id = abc_id_cli')
            ->group_by('cli_id');
        if ($where != null) {
            $this->db->where($where);
        }
        $array = $query->get()->result_array();
        return $array;
    }

    /**
     * @author Raul Escalante
     * @since 2019-5-18
     * @return reporte de la composicion del saldo por clientes
     * @param $where array para condicionar.
     */
    public function get_composicion_saldo($fecha_hasta, $where = null, $having = null)
    {
        $tipo_factura = json_decode($this->gsatelite->get_parametro('SYS_TIPO_FACTURAS'), true);
        $not_cred = (isset($tipo_factura['NOTA CREDITO'])) ? $tipo_factura['NOTA CREDITO'] : 0;
        $not_remi = (isset($tipo_factura['NOTA REMISION'])) ? $tipo_factura['NOTA REMISION'] : 0;
        $excluidas = array($not_cred, $not_remi);
        $sql = "ven_id, CONCAT_WS(' ', ven_nombre, ven_apellidos) AS ven_nombre, cli_id, cli_codigo, fac_total_retencion,
                IF(cli_personeria = 1, CONCAT_WS(' ', cli_nombre, cli_apellido), cli_apellido) AS cli_nombre, suc_codigo, suc_nombre,
                tif_nombre, tif_abreviacion, fac_numero, fac_id, fac_fecha_fact, fac_fecha_vencimiento,fac_id_origen_devolucion, fac_monto_abono, 
                IF(((fac_fecha_vencimiento IS NOT NULL) AND (fac_fecha_fact IS NOT NULL)), (DATEDIFF(fac_fecha_vencimiento, fac_fecha_fact)), '') AS plazo,
                IF((fac_fecha_vencimiento IS NOT NULL), (DATEDIFF(CURDATE(), fac_fecha_vencimiento)), '') dias_vencido, 
                (fac_total - fac_total_retencion) AS fac_total, fac_monto_pendiente, 
                IF(fac_total_retencion > 0, (fac_total_retencion + IFNULL((SELECT SUM(abf_monto) FROM abf_abono_factura JOIN abc_abono_cliente ON abc_id = abf_id_abc 
                    WHERE abf_id_fac = fac_id AND DATE(abc_fecha) <= '$fecha_hasta' AND abc_estado = 1), 0)), IFNULL((SELECT SUM(abf_monto) FROM abf_abono_factura 
                    JOIN abc_abono_cliente ON abc_id = abf_id_abc WHERE abf_id_fac = fac_id AND DATE(abc_fecha) <= '$fecha_hasta' AND abc_estado = 1), 0)) AS abonos_hasta,
                (fac_total - fac_total_retencion - IFNULL((SELECT SUM(abf_monto) FROM abf_abono_factura JOIN abc_abono_cliente ON abc_id = abf_id_abc
                    WHERE abf_id_fac = fac_id AND DATE(abc_fecha) <= '$fecha_hasta' AND abc_estado = 1), 0)) AS saldo_actual";
        $query = $this->db->select($sql, false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'LEFT')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'LEFT')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'LEFT')
            ->join('suc_sucursal', 'suc_id = fac_id_suc', 'LEFT')
            ->where('DATE(fac_fecha_fact)  <=', $fecha_hasta)
            ->where_not_in('fac_id_tif', $excluidas)
            ->group_by('fac_id')
            ->order_by('ven_id, cli_id, fac_fecha_fact, fac_numero');
        if ($where != null) {
            $this->db->where($where);
        }
        if ($having != null) {
            $this->db->having($having);
        }
        $array = $query->get()->result_array();
        $sql = $this->db->last_query();
        return $array;
    }

    /**
     * @author Raul Escalante
     * @since 06/09/2019
     * @return array con los datos del pago
     */
    public function get_datos_recibo($where = null, $group_by = null)
    {
        $query = $this->db->select("*, 
            IFNULL((
                SELECT GROUP_CONCAT(CONCAT(frp_nombre, IFNULL((SELECT CONCAT(' (', ban_nombre, ')') FROM ban_banco WHERE ban_id = dab_id_ban), '')) ORDER BY frp_id ASC SEPARATOR ', ')
                FROM dab_detalle_abono INNER JOIN frp_forma_pago ON frp_id = dab_id_frp WHERE dab_id_abc = abc_id),
            '') AS formas_de_pago
            ", false)
            ->from('abc_abono_cliente')
            ->join('abf_abono_factura', 'abf_id_abc = abc_id')
            ->join('fac_factura', 'fac_id = abf_id_fac')
            ->join('cli_cliente', 'cli_id = abc_id_cli')
            ->join('dab_detalle_abono', 'dab_id_abc = abc_id', 'left')
            ->join('cmr_comprobante_retencion', 'cmr_id_abf = abf_id', 'left')
            ->join('anl_anticipo_cliente', 'anl_id_abc_origen = abc_id', 'left');
        if ($where) {
            $this->db->where($where);
        }

        if ($group_by != null) {
            $this->db->group_by($group_by);
        }

        return $query->get()->result_array();
    }

    /**
     * @author Raul Escalante
     * @since Oct/2020
     * @return array con los datos del anticipo
     */
    public function get_datos_recibo_anticipo($where = null, $group_by = null)
    {
        $query = $this->db->select("*,
            IFNULL((SELECT GROUP_CONCAT(ban_nombre) FROM dab_detalle_abono LEFT JOIN ban_banco ON ban_id = dab_id_ban WHERE dab_id_anl = anl_id), ' - ') AS banco_nom,
            IFNULL((SELECT GROUP_CONCAT(frp_nombre) FROM frp_forma_pago INNER JOIN dab_detalle_abono ON dab_id_frp = frp_id WHERE dab_id_anl = anl_id), ' - ') AS forma_pago,
            IFNULL((SELECT GROUP_CONCAT(frp_nombre) FROM frp_forma_pago INNER JOIN dab_detalle_abono ON dab_id_frp = frp_id WHERE dab_id_anl = anl_id), ' - ') AS formas_de_pago
            ", false)
            ->from('anl_anticipo_cliente')
            ->join('cli_cliente', 'cli_id = anl_id_cli')
            ->join('abc_abono_cliente', 'abc_id = anl_id_abc_origen', 'left')
            ->join('dab_detalle_abono', 'dab_id_anl = anl_id', 'left');
        if ($where) {
            $this->db->where($where);
        }
        return $query->get()->result_array();
    }

    public function get_cuentas_antiguedad_saldo_corto_plazo($where = null, $having = null, $desde = "", $hasta = "")
    {
        $fecha_sin_vencer   = "";
        $fecha_rango1       = "";
        $fecha_rango2       = "";
        $fecha_rango3       = "";
        if ($desde != "" && $hasta != "") {
            $fecha_sin_vencer   = " AND DATE(f1.fac_fecha_fact) >= '$desde' AND  DATE(f1.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango1       = " AND DATE(f2.fac_fecha_fact) >= '$desde' AND  DATE(f2.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango2       = " AND DATE(f3.fac_fecha_fact) >= '$desde' AND  DATE(f3.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango3       = " AND DATE(f4.fac_fecha_fact) >= '$desde' AND  DATE(f4.fac_fecha_fact) <= '$hasta' ";
        }
        $this->db->select("cli_codigo, IF(cli_personeria=1, CONCAT_WS(' ', cli_nombre, cli_apellido), cli_apellido) AS cliente,
                            GROUP_CONCAT( DISTINCT CONCAT_WS(' ', ven_nombre, ven_apellidos)) AS vendedor, 
                            (SELECT SUM(f1.fac_monto_pendiente) from fac_factura f1 WHERE f1.fac_id_cli = cli_id and f1.fac_estado = 1 
                            and  DATEDIFF(CURDATE(),f1.fac_fecha_vencimiento) <=0 $fecha_sin_vencer) AS sin_vencer, 
                            (SELECT SUM(f2.fac_monto_pendiente) from fac_factura f2 WHERE f2.fac_id_cli = cli_id and f2.fac_estado = 1
                            and DATEDIFF(CURDATE(),f2.fac_fecha_vencimiento) BETWEEN  1 AND 7 $fecha_rango1) AS rango1, 
                            (SELECT SUM(f3.fac_monto_pendiente) from fac_factura f3 WHERE f3.fac_id_cli = cli_id and f3.fac_estado = 1 
                            and  DATEDIFF(CURDATE(),f3.fac_fecha_vencimiento) BETWEEN  8 AND 15 $fecha_rango2) AS rango2, 
                            (SELECT SUM(f4.fac_monto_pendiente) from fac_factura f4 WHERE f4.fac_id_cli = cli_id and f4.fac_estado = 1 
                            and DATEDIFF(CURDATE(),f4.fac_fecha_vencimiento) BETWEEN  16 AND 30 $fecha_rango3) AS rango3, 
                            scl_codigo, scl_nombre", false)
            ->from('fac_factura')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('scl_sucursal_cliente', 'scl_id = fac_id_scl', 'left')
            ->group_by('fac_id_cli');
        if ($where) {
            $this->db->where($where);
        }
        if ($having) {
            $this->db->having($having, false);
        }
        $datos = $this->db->get()->result_array();
        return $datos;
    }

    /**
     * @author Marlon Olmedo
     * @since 2019-12-27
     * @return Datos de los anticipos
     * @param $where array asociativo para condicionar
     */
    public function get_anticipo_cliente($where = null)
    {
        $this->db->select('*, IF(cli_personeria = 1, CONCAT( cli_nombre, " ", cli_apellido ), cli_apellido)  AS cliente,
                IFNULL((SELECT GROUP_CONCAT(fac_numero) FROM fac_factura INNER JOIN abf_abono_factura ON abf_id_fac = fac_id 
                WHERE abf_id_abc = anl_id_abc_origen AND fac_estado != 0 ),"-") AS facturas
            ', false)
            ->from('anl_anticipo_cliente')
            ->join('cli_cliente', 'cli_id = anl_id_cli');
        if ($where) {
            $this->db->where($where);
        }

        return $this->db->get()->result_array();
    }


    /**
     * @author Mario Marquez
     * @since 2020-09-28
     * @return Datos de las facturas y sus saldos agrupados por cliente condicionando por rango de dias especificos.
     * @param $where array asociativo para condicionar
     */
    public function get_cuentas_antiguedad_saldo_detallado($where = null, $having = null, $desde = "", $hasta = "")
    {
        $fecha_sin_vencer   = "";
        $fecha_rango1       = "";
        $fecha_rango2       = "";
        $fecha_rango3       = "";
        $fecha_rango4       = "";
        $fecha_restantes    = "";
        if ($desde != "" && $hasta != "") {
            $fecha_sin_vencer   = " AND DATE(f1.fac_fecha_fact) >= '$desde' AND  DATE(f1.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango1       = " AND DATE(f2.fac_fecha_fact) >= '$desde' AND  DATE(f2.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango2       = " AND DATE(f3.fac_fecha_fact) >= '$desde' AND  DATE(f3.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango3       = " AND DATE(f4.fac_fecha_fact) >= '$desde' AND  DATE(f4.fac_fecha_fact) <= '$hasta' ";
            $fecha_rango4       = " AND DATE(f5.fac_fecha_fact) >= '$desde' AND  DATE(f5.fac_fecha_fact) <= '$hasta' ";
            $fecha_restantes    = " AND DATE(f7.fac_fecha_fact) >= '$desde' AND  DATE(f7.fac_fecha_fact) <= '$hasta' ";
        }
        $this->db->select("(fac_id) numero_facturas,
                    fac_numero,
                    fac_fecha_fact,
                    fac_fecha_vencimiento,
                    cli_id, cli_codigo,
                    IF(cli_personeria=1, CONCAT_WS(' ', cli_nombre, cli_apellido), cli_apellido) cliente,
                    ((fac_total - fac_total_retencion)) AS fac_total, 
                    (fac_monto_abono) AS fac_monto_abono, 
                    ((fac_monto_pendiente - fac_total_retencion)) AS fac_monto_pendiente,
                    IFNULL((
                        SELECT SUM((f1.fac_monto_pendiente - f1.fac_total_retencion)) from fac_factura f1
                        WHERE f1.fac_id_cli = cli_id and f1.fac_estado = 1 and  DATEDIFF(CURDATE(),f1.fac_fecha_vencimiento)<=0 $fecha_sin_vencer
                        AND f1.fac_id = fac.fac_id
                        )
                    , 0) AS sin_vencer, 
                    IFNULL((
                        SELECT SUM((f2.fac_monto_pendiente - f2.fac_total_retencion)) from fac_factura f2
                        WHERE f2.fac_id_cli = cli_id and f2.fac_estado = 1 and DATEDIFF(CURDATE(),f2.fac_fecha_vencimiento) BETWEEN  1 AND 30 $fecha_rango1
                        AND f2.fac_id = fac.fac_id
                        )
                    , 0) AS rango1,
                    IFNULL((
                        SELECT SUM((f3.fac_monto_pendiente - f3.fac_total_retencion)) from fac_factura f3
                        WHERE f3.fac_id_cli = cli_id and f3.fac_estado = 1 and  DATEDIFF(CURDATE(),f3.fac_fecha_vencimiento) BETWEEN  31 AND 60 $fecha_rango2
                        AND f3.fac_id = fac.fac_id
                        )
                    , 0) AS rango2, 
                    IFNULL((
                        SELECT SUM((f4.fac_monto_pendiente - f4.fac_total_retencion)) from fac_factura f4
                        WHERE f4.fac_id_cli = cli_id and f4.fac_estado = 1 and DATEDIFF(CURDATE(),f4.fac_fecha_vencimiento) BETWEEN  61 AND 90 $fecha_rango3
                        AND f4.fac_id = fac.fac_id
                        )
                    , 0) AS rango3, 
                    IFNULL((
                        SELECT SUM((f5.fac_monto_pendiente - f5.fac_total_retencion)) from fac_factura f5
                        WHERE f5.fac_id_cli = cli_id and f5.fac_estado = 1 and DATEDIFF(CURDATE(),f5.fac_fecha_vencimiento) BETWEEN  91 AND 120 $fecha_rango4
                        AND f5.fac_id = fac.fac_id
                        )
                    , 0) AS rango4,
                    IFNULL((
                        SELECT SUM((f7.fac_monto_pendiente - f7.fac_total_retencion)) from fac_factura f7
                        WHERE f7.fac_id_cli = cli_id and f7.fac_estado = 1 and DATEDIFF(CURDATE(),f7.fac_fecha_vencimiento) > 120 $fecha_restantes
                        AND f7.fac_id = fac.fac_id
                        ) 
                    , 0) AS restantes,
                     scl_codigo, scl_nombre
                     ", false)
            ->from('fac_factura as fac')
            ->join('cli_cliente', 'cli_id = fac_id_cli', 'left')
            ->join('ven_vendedor', 'ven_id = fac_id_ven', 'left')
            ->join('tif_tipo_factura', 'tif_id = fac_id_tif', 'left')
            ->join('scl_sucursal_cliente', 'scl_id = fac_id_scl', 'left')
            //->group_by('fac_id_cli')
        ;
        if ($where) {
            $this->db->where($where);
        }

        if ($having) {
            $this->db->having($having, false);
        }

        $datos = $this->db->get()->result_array();
        return $datos;
    }
}
