<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With*');
header("Content-Type: application/json; charset=UTF-8");


/** 
 * V1.php
 * Descripción:
 * Web service para gestión de envío y recepción de informacion de solicitudes de activo
 * @package Defensoria
 * @subpackage Modulo - Activo Fijo
 * @copyright Copyright (c) 2025 Grupo Satélite S.A. DE C.V.
 * @author Eduardo Mejia
 * @version 2024 - 01
 * @since Archivo disponible desde 2025-03-03
 */

if(!defined('BASEPATH'))
	exit('No direct script access allowed');

class V1 extends CI_Controller
{

  private $userData;

	/**
   * Descripción:
   * Función index del controlador
   * @author -----
   * @since 2024-01-05
   */
	public function __construct() {
    parent::__construct();
    $this->ci =& get_instance();

    $this->middlewareAuthorization();

    $this->load->model('gsatelite');
    $this->load->model('activo_fijo');
    $this->load->model('sqlserver_model');
    $this->load->library('pdf');


  }

  /**
   * Descripción:
   * Función para estandarizar el formato de la respuesta json
   * @author -----
   * @since 2024-01-05
   */
  public function response_json($data = null, $status = true, $message = 'ok', $httpCode = 200) {

    $response = array(
      'data' => null,
      'status' => $status,
      'message' => $message,
    );

    if($data){
      $response['data'] = $data;
    }

    // Limpiar el búfer si está activo
    if (ob_get_level() > 0) {
        ob_end_clean();
    }

    http_response_code($httpCode);
    die(json_encode($response, \JSON_UNESCAPED_UNICODE));
  } 

  /**
   * Descripción:
   * Parsear body del request
   * @author -----
   * @since 2024-01-05
   */
  private function file_get_contents_utf8($fn) {
   $content = file_get_contents($fn);
   return mb_convert_encoding($content, 'UTF-8',
    mb_detect_encoding($content, 'UTF-8, ISO-8859-1', true));
 } 

   /**
   * Descripción:
   * Verifica api-key en la solicitud
   * @author -----
   * @since 2024-01-05
   */
   private function middlewareAuthorization() {
    if(!function_exists('getallheaders')) {
      $this->response_json(null, false, "Verificar servidor, imposible validar");
      if (strpos(PHP_SAPI, 'cgi') === 0) {
        header('Status: 500 ', TRUE);
      } else {
        $server_protocol = (isset($_SERVER['SERVER_PROTOCOL']) && in_array($_SERVER['SERVER_PROTOCOL'], array('HTTP/1.0', 'HTTP/1.1', 'HTTP/2'), TRUE))
        ? $_SERVER['SERVER_PROTOCOL'] : 'HTTP/1.1';
        header($server_protocol.' 500', TRUE, '500');
      }
      die(json_encode($respuesta, \JSON_UNESCAPED_UNICODE));
    } 

    $cabeceras_server = getallheaders();
    if(!isset($cabeceras_server['Authorization']) || (isset($cabeceras_server['Authorization']) && trim($cabeceras_server['Authorization']) == '')) {
      $this->response_json(null, false, "Debe proporcionar un API-KEY para autenticación", 401);          
    }
    else if(isset($cabeceras_server['Authorization'])  && trim($cabeceras_server['Authorization']) != '') {
      $Authorization = trim($cabeceras_server['Authorization']);
      $API_KEY = $this->gsatelite->get_parametro('API_KEY_ACTIVO_FIJO');
      if(trim($API_KEY) == '' or strlen(trim($API_KEY)) < 32) {
        $this->response_json(null, false, "Solicitar la configuración de un API-KEY valida", 401);          
      }
      else if(trim($API_KEY) != trim($Authorization)  ) {
        $this->response_json(null, false, "La clave API que has proporcionado no es válida. Por favor, verifica la clave e intenta nuevamente.", 401);          
      }
    }    
  }

   /**
   * Descripción:
   * Verifica el header Content-Type
   * @author -----
   * @since 2024-01-05
   */  
   private function middlewareContentType() {
    $headers = getallheaders();
    $contentType = $headers["Content-Type"];
    if(!isset($contentType) || (isset($contentType) && trim($contentType) == '') || $contentType !== 'application/json') {
      return $this->response_json(null, false, "El encabezado 'Content-Type' debe tener el valor 'application/json' para procesar la solicitud correctamente.", 415);
    }
  }

  /**
   * Descripción:
   * Función para validar el metodo http permitido para la solicitud 
   * @author -----
   * @since 2024-01-05
   * @return string jwt
   */  
  private function validarMetodoHttp($metodoPermitido) {
    $http_method = $this->input->server('REQUEST_METHOD');
    if ($http_method !== $metodoPermitido) {
      return $this->response_json(null, false, "Método HTTP no permitido", 405);
    }
  }

  /**
   * Descripción:
   * Solicitudes por empleado
   * @author Eduardo Mejia - Grupo Satelite
   * @since 2024-01-05
   * @param int $emp_codigo
   * @return solicitudes y activos x solicitud
   */  
  public function obtener_solicitudes($emp_codigo = 0) {
    $this->validarMetodoHttp("GET");
    $data = [];

    if ($emp_codigo == 0) {
      $message = "Id empleado requerido";
      $this->response_json($data, false, $message);
    }

    $empleado = $this->gsatelite->get_tabla('emp_empleado', ['emp_codigo' => $emp_codigo]);
    if (empty($empleado)) {
      // Insertar empleado;
      $emp_codigo = $this->insertarEmpleadoSinco($emp_codigo);
    }    

    $this->db->join('emp_empleado', 'emp_id = slt_id_emp');
    $solicitudes = $this->gsatelite->get_tabla('slt_solicitud_activos', ['emp_codigo' => $emp_codigo, 'slt_estado' => 1]);
    $response = [];

    foreach ($solicitudes as $slt) {
      $solicitud = [
        'fecha_solicitud' => $slt['slt_fecha'],
        'id_solicitud' => $slt['slt_id'],
        'activos' => [],
      ];

      $where = [
        'axs_id_slt' => $slt['slt_id'], 
      ];  

      $activos = $this->activo_fijo->obtenerActivosxSolicitud($where);
      $solicitud['activos'] = $activos;
      $response[] = $solicitud;
    }

    $data['solicitudes'] = $response;

    return $this->response_json($data);
  }

  /**
   * Descripción:
   * Aprobar / Rechazar solicitud de activo fijo 
   * @author Eduardo Mejia - Grupo Satelite
   * @since 2024-01-05
   * @param int $idUsuario
   * @return array informacion asociada al usuario
   */  
  public function gestionar_solicitud() {
    $this->validarMetodoHttp("POST");
    $data = [];

    $content = $this->file_get_contents_utf8("php://input");
    $request = json_decode($content, true); 

    if (!isset($request['id_solicitud'])) {
      return $this->response_json(null, false, "El campo id_solicitud es requerido", 400);      
    }   

    if (!isset($request['estado'])) {
      return $this->response_json(null, false, "El campo estado es requerido", 400);      
    }   

    if (!isset($request['observaciones'])) {
      $request['observaciones'] = NULL;
    }   

    if (!in_array($request['estado'], [2, 3])) {
      return $this->response_json(null, false, "El estado puede ser 2 = Aprobado o 3 = Rechazado", 400);      
    }    

    $solicitud = $this->gsatelite->get_tabla('slt_solicitud_activos', ['slt_id' => $request['id_solicitud']]);

    if (empty($solicitud)) {
      return $this->response_json(null, false, "No se encontró el registro", 404);      
    } 

    if ($solicitud[0]['slt_estado'] != 1) {
      return $this->response_json(null, true, "La solicitud ya ha sido aprobada previamente. No es necesario aprobarla nuevamente.", 409);      
    }

    $estado = $request['estado'];
    $observaciones = $request['observaciones'];
    $id_registro = $request['id_solicitud'];
    $id_usuario = 1;

    if ($estado == 2) { //Aprobado
        $actualizar_slt = array(
            'slt_usu_aprueba' => $id_usuario,
            'slt_fecha_aprobacion' => date('Y-m-d H:i:s'),
            'slt_estado' => 2, 
            'slt_usu_modifica' => $id_usuario,
            'slt_fecha_modifica' => date('Y-m-d H:i:s')
        );
        $registro = $this->gsatelite->actualizar_array('slt_solicitud_activos', $actualizar_slt, array('slt_id' => $id_registro));

                        $this->db->join('slt_solicitud_activos', 'slt_id = axs_id_slt');
                        $this->db->join('act_activo_fijo', 'act_id = axs_id_act');
                        $this->db->join('emp_empleado', 'emp_id = slt_id_emp');
        $tabla_detail = $this->gsatelite->get_tabla('axs_activo_solicitud', array('axs_id_slt' => $id_registro));
        foreach ($tabla_detail as $k => $v) {
            if ($v['slt_tipo_solicitud'] == 1) { // Traslado
                $data_update=array(
                    'axe_estado'=> 0,
                    'axe_use_mod'=> $id_usuario,
                    'axe_fecha_mod'=>date('Y-m-d H:i:s')
                );
                $this->db->where('axe_id_act',$v['act_id']);
                $this->db->where('axe_estado',1);
                $this->db->update('axe_activoxempleado',$data_update);
            }
            $axerow = array(
                'axe_id_act' => $v['act_id'],
                'axe_id_emp' => $v['slt_id_emp'],
                'axe_id_are' => $v['slt_id_are'],
                'axe_fecha'  => date('Y-m-d H:i:s'),
                'axe_obs'    => $observaciones,
                'axe_use_mod' => $id_usuario,
                'axe_fecha_mod' => date('Y-m-d H:i:s')
            );
            $nuevo = $this->db->insert('axe_activoxempleado', $axerow);
            if ($v['slt_tipo_solicitud'] == 1) {
                $actualiza = array(
                    "act_id_are" => $v['slt_id_are'],
                    "act_usu_modificacion" => $id_usuario,
                    "act_fecha_modificacion" => date("Y-m-d H:i:s")
                );
                $resp = $this->gsatelite->actualizar_array('act_activo_fijo', $actualiza, array('act_id' => $v['act_id']));
            }
        }
    } else if($estado == 3) { //Rechazado
        $actualizar_slt = array(
            'slt_usu_rechaza' => $id_usuario,
            'slt_fecha_denegado' => date('Y-m-d H:i:s'),
            'slt_estado' => 0, 
            'slt_usu_modifica' => $id_usuario,
            'slt_fecha_modifica' => date('Y-m-d H:i:s')
        );
        $registro = $this->gsatelite->actualizar_array('slt_solicitud_activos', $actualizar_slt, array('slt_id' => $id_registro));                
    }

    $message = "Solicitud " . (($estado == 2) ? 'Aprobada' : "Rechazada") . " correctamente. ";
    return $this->response_json(null, true, $message, 201);      

  }  

  /**
   * Descripción:
   * Historial Solicitudes por empleado
   * @author Eduardo Mejia - Grupo Satelite
   * @since 2024-01-05
   * @param int $emp_codigo
   * @return Historial solicitudes y activos x solicitud
   */  
  public function historial_solicitudes($emp_codigo = 0) {
    $this->validarMetodoHttp("GET");
    $data = [];

    if ($emp_codigo == 0) {
      $message = "Id empleado requerido";
    $this->response_json($data, false, $message);
    }

    $this->db->join('emp_empleado', 'emp_id = slt_id_emp');
    $solicitudes = $this->gsatelite->get_tabla('slt_solicitud_activos', ['emp_codigo' => $emp_codigo, 'slt_estado != 1' => null]);
    $response = [];

    foreach ($solicitudes as $slt) {
      $solicitud = [
        'fecha_solicitud' => $slt['slt_fecha'],
        'estado_solicitud' => $this->estado_solicitud($slt['slt_estado']),
        'fecha_aprobacion' => $slt['slt_fecha_aprobacion'],
        'fecha_rechazo' => $slt['slt_fecha_denegado'],
        'activos' => [],
      ];

      $where = [
        'axs_id_slt' => $slt['slt_id'], 
      ];  

      $activos = $this->activo_fijo->obtenerActivosxSolicitud($where);
      $solicitud['activos'] = $activos;
      $response[] = $solicitud;
    }

    $data['solicitudes'] = $response;

    return $this->response_json($data);
  } 

  private function estado_solicitud($slt_estado) {
    $estados = [
      0 => "Rechazada",
      1 => "Pendiente",
      2 => "Aprobada",
    ];
    return isset($estados[$slt_estado]) ? $estados[$slt_estado] : "Desconocido";
  } 

  function acta_asignacion($emp_codigo = 0){

    // Obteniendo empleado

    if ($emp_codigo == 0) {
      return $this->response_json(null, false, "Id empleado requerido");      
    }

    $empleado = $this->gsatelite->get_tabla('emp_empleado', ['emp_codigo' => $emp_codigo]);

    if (empty($empleado)) {
      return $this->response_json(null, false, "No se ha encontrado un empleado con el código proporcionado.", 404);      
    }

    $emp_id = $empleado[0]['emp_id'];

    $where['emp_id'] = $emp_id;
    $where['axe_estado'] = 1;
    $detalles = $this->activo_fijo->get_reporte_asignacion($where);
    $act=array();
    foreach ($detalles as $det) {
      $act[]= $det['axe_id_act'];
    }

    if (empty($act)) {
      $act = [0];
    }

    $activos_data = $this->activo_fijo->activo_report($act,$emp_id);

    if (empty($activos_data)) {
      return $this->response_json(null, false, "No se ha encontrado activos asignados al empleado.", 404);      
    }

    $GLOBALS['nombre_reporte'] = 'Acta de Asignación de Activos Fijos por Responsable ' . date('dmY');
    $reporte = $this->load->view('activo/nuevo/acta_reporte_defensoria', array('activos' => $activos_data), true);
    $pdf_output = $this->pdf->makePdf2($reporte, 'L', 'acta_de_asignacion', 'Letter', 'S');
    $pdf_base64 = base64_encode($pdf_output);


    if (empty($activos_data)) {
      return $this->response_json(null, false, "No se ha encontrado activos asignados al empleado.", 404);      
    }    
    
    $data['pdf_base64'] = $pdf_base64;
    return $this->response_json($data);      
  }

  public function getEmpleados($codigoEmpleadoSinco = null) {
    ob_end_clean();
    echo json_encode($this->sqlserver_model->getEmpleadosSinco($codigoEmpleadoSinco), JSON_UNESCAPED_UNICODE);
    die();
  }

  private function insertarEmpleadoSinco($emp_codigo){
    $new_emp_codigo = 0;
    $empleadoSinco = $this->sqlserver_model->getEmpleadosSinco($emp_codigo);
    if (!empty($empleadoSinco)) {
      $empleadoSinco = $empleadoSinco[0];
      $codigoArea = $empleadoSinco['IDDIRECCION']; // TODO: Verificar si se tomara IDDIRECCIN O IDSUBDIRECCION
      $codigoArea = str_pad($codigoArea, 2, "0", STR_PAD_LEFT);
      $area = $this->gsatelite->get_tabla('are_area', ['are_prefijo' => $codigoArea]);
      if (!empty($area)) {
        $id_usuario = 1;
        $area = $area[0];
        $insert = [
          'emp_codigo' => $empleadoSinco['IDEMPLEADO'],
          'emp_nombre' => $empleadoSinco['NOMBREEMPLEADO'],
          'emp_apellido' => $empleadoSinco['APELLIDOEMPLEADO'],
          'emp_email' => $empleadoSinco['CORREO'],
          'emp_id_are' => $area['are_id'],
          'emp_id_suc' => $area['are_id_suc'],
          'emp_estado' => 1,
          'emp_fecha_mod' => date('Y-m-d H:i:s'),
          'emp_fecha_crea' => date('Y-m-d H:i:s'),
          'emp_usu_mod' => $id_usuario,
          'emp_usu_crea' => $id_usuario,
        ];

        $emp_id = $this->gsatelite->ingresar('emp_empleado', $insert);
        $new_empleado = $this->gsatelite->get_tabla('emp_empleado', ['emp_id' => $emp_id]);
        if (!empty($new_empleado)) {
          $new_emp_codigo = $new_empleado[0]['emp_codigo'];
        }
      }
    }
    return $new_emp_codigo;

  }   



}